<?php
/**
 * Test script to verify the updated order confirmation payload format
 * Place this file in your WordPress root directory and access via browser
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    // Load WordPress
    require_once('wp-config.php');
    require_once('wp-load.php');
}

// Check if WooCommerce is active
if (!class_exists('WooCommerce')) {
    die('WooCommerce is not active');
}

echo "<h1>Loadlink Order Confirmation Payload Test</h1>";

// 1. Test payload building
echo "<h2>1. Payload Format Test</h2>";

if (class_exists('Loadlink_Client')) {
    echo "✅ Loadlink_Client class exists<br>";
    
    // Create a test order
    $test_order = wc_create_order();
    if ($test_order) {
        echo "✅ Test order created: #" . $test_order->get_id() . "<br>";
        
        // Set up test data
        $test_order->set_billing_first_name('John');
        $test_order->set_billing_last_name('Doe');
        $test_order->set_billing_email('john.doe@test.com');
        $test_order->set_billing_phone('0401234567');
        $test_order->set_billing_address_1('123 Test Street');
        $test_order->set_billing_city('Test City');
        $test_order->set_billing_state('NSW');
        $test_order->set_billing_postcode('2000');
        $test_order->set_billing_country('AU');
        
        $test_order->set_shipping_first_name('John');
        $test_order->set_shipping_last_name('Doe');
        $test_order->set_shipping_address_1('456 Delivery Street');
        $test_order->set_shipping_city('Delivery City');
        $test_order->set_shipping_state('VIC');
        $test_order->set_shipping_postcode('3000');
        $test_order->set_shipping_country('AU');
        
        $test_order->save();
        
        // Test payload building
        $client = new Loadlink_Client();
        
        // Mock order data
        $order_data = [
            'order' => $test_order,
            'pickup_address' => [
                'state' => 'NSW',
                'suburb' => 'SYDNEY',
                'postcode' => '2000',
                'address1' => '123 Store Street',
                'address2' => '',
                'address3' => '',
                'company_name' => 'Test Store',
                'contact_name' => 'Store Manager',
                'email' => 'store@test.com',
                'phone_area_code' => '02',
                'phone' => '9876543210'
            ],
            'dropoff_address' => [
                'state' => 'VIC',
                'suburb' => 'MELBOURNE',
                'postcode' => '3000',
                'address1' => '456 Customer Street',
                'address2' => 'Unit 1',
                'address3' => '',
                'company_name' => '',
                'contact_name' => 'John Doe',
                'email' => 'john.doe@test.com',
                'phone_area_code' => '03',
                'phone' => '0401234567'
            ]
        ];
        
        // Test service quote ID
        $service_quote_id = 12345;
        
        echo "<h3>Testing Payload Building</h3>";
        
        // Use reflection to test private method
        $reflection = new ReflectionClass($client);
        $method = $reflection->getMethod('build_confirmation_payload');
        $method->setAccessible(true);
        
        try {
            $payload = $method->invoke($client, $service_quote_id, $order_data);
            
            echo "✅ Payload built successfully<br>";
            
            // Check payload structure
            echo "<h3>Payload Structure Validation</h3>";
            
            $required_fields = [
                'freight_service_quote_id',
                'external_order_id',
                'external_order_no',
                'pickup_state',
                'pickup_suburb',
                'pickup_postcode',
                'pickup_address1',
                'pickup_address2',
                'pickup_address3',
                'pickup_company_name',
                'pickup_company_contact_name',
                'pickup_company_email',
                'pickup_company_phone_area_code',
                'pickup_company_phone',
                'dropoff_state',
                'dropoff_suburb',
                'dropoff_postcode',
                'dropoff_address1',
                'dropoff_address2',
                'dropoff_address3',
                'dropoff_company_name',
                'dropoff_company_contact_name',
                'dropoff_company_email',
                'dropoff_company_phone_area_code',
                'dropoff_company_phone'
            ];
            
            $missing_fields = [];
            foreach ($required_fields as $field) {
                if (!isset($payload[$field])) {
                    $missing_fields[] = $field;
                }
            }
            
            if (empty($missing_fields)) {
                echo "✅ All required fields present<br>";
            } else {
                echo "❌ Missing fields: " . implode(', ', $missing_fields) . "<br>";
            }
            
            // Check that freight_id is NOT present
            if (!isset($payload['freight_id'])) {
                echo "✅ freight_id correctly removed from payload<br>";
            } else {
                echo "❌ freight_id still present in payload<br>";
            }
            
            // Check data types
            echo "<h3>Data Type Validation</h3>";
            
            $type_checks = [
                'freight_service_quote_id' => 'integer',
                'external_order_id' => 'string',
                'external_order_no' => 'string',
                'pickup_state' => 'string',
                'dropoff_state' => 'string'
            ];
            
            foreach ($type_checks as $field => $expected_type) {
                if (isset($payload[$field])) {
                    $actual_type = gettype($payload[$field]);
                    if ($actual_type === $expected_type) {
                        echo "✅ $field is $actual_type<br>";
                    } else {
                        echo "❌ $field is $actual_type (expected $expected_type)<br>";
                    }
                }
            }
            
            // Display the payload
            echo "<h3>Generated Payload</h3>";
            echo "<pre style='background: #f0f0f0; padding: 10px; max-height: 400px; overflow-y: auto;'>";
            echo htmlspecialchars(json_encode($payload, JSON_PRETTY_PRINT));
            echo "</pre>";
            
            // Compare with reference format
            echo "<h3>Reference Format Comparison</h3>";
            $reference_payload = [
                "freight_service_quote_id" => 1070,
                "external_order_id" => "PD001",
                "external_order_no" => "OD001",
                "pickup_state" => "QLD",
                "pickup_suburb" => "SOUTHPORT",
                "pickup_postcode" => "4215",
                "pickup_address1" => "36 Johnston St",
                "pickup_address2" => "8 kaertl str",
                "pickup_address3" => "",
                "pickup_company_name" => "",
                "pickup_company_contact_name" => "Peter Web",
                "pickup_company_email" => "peter.web@test.com",
                "pickup_company_phone_area_code" => "44",
                "pickup_company_phone" => "0401010101",
                "dropoff_state" => "QLD",
                "dropoff_suburb" => "PARADISE POINT",
                "dropoff_postcode" => "4216",
                "dropoff_address1" => "1779 Murbko Rd",
                "dropoff_address2" => null,
                "dropoff_address3" => "",
                "dropoff_company_name" => "",
                "dropoff_company_contact_name" => "Church State",
                "dropoff_company_email" => "church@test.com",
                "dropoff_company_phone_area_code" => "44",
                "dropoff_company_phone" => "0403030303"
            ];
            
            echo "<h4>Reference Payload Structure:</h4>";
            echo "<pre style='background: #e8f4fd; padding: 10px; max-height: 300px; overflow-y: auto;'>";
            echo htmlspecialchars(json_encode($reference_payload, JSON_PRETTY_PRINT));
            echo "</pre>";
            
            // Check structure match
            $reference_keys = array_keys($reference_payload);
            $generated_keys = array_keys($payload);
            
            $missing_in_generated = array_diff($reference_keys, $generated_keys);
            $extra_in_generated = array_diff($generated_keys, $reference_keys);
            
            if (empty($missing_in_generated) && empty($extra_in_generated)) {
                echo "✅ Payload structure matches reference format perfectly<br>";
            } else {
                if (!empty($missing_in_generated)) {
                    echo "❌ Missing fields in generated payload: " . implode(', ', $missing_in_generated) . "<br>";
                }
                if (!empty($extra_in_generated)) {
                    echo "❌ Extra fields in generated payload: " . implode(', ', $extra_in_generated) . "<br>";
                }
            }
            
        } catch (Exception $e) {
            echo "❌ Error building payload: " . $e->getMessage() . "<br>";
        }
        
        // Clean up test order
        $test_order->delete(true);
        echo "✅ Test order cleaned up<br>";
        
    } else {
        echo "❌ Could not create test order<br>";
    }
} else {
    echo "❌ Loadlink_Client class not found<br>";
}

// 2. Test method signatures
echo "<h2>2. Method Signature Test</h2>";

if (class_exists('Loadlink_Client')) {
    $client = new Loadlink_Client();
    
    // Check confirm_freight_order method
    $reflection = new ReflectionClass($client);
    $method = $reflection->getMethod('confirm_freight_order');
    $parameters = $method->getParameters();
    
    echo "confirm_freight_order method parameters:<br>";
    foreach ($parameters as $param) {
        echo "- " . $param->getName() . " (" . ($param->getType() ? $param->getType() : 'mixed') . ")<br>";
    }
    
    if (count($parameters) === 2) {
        echo "✅ Method signature updated correctly (2 parameters)<br>";
    } else {
        echo "❌ Method signature has " . count($parameters) . " parameters (expected 2)<br>";
    }
}

// 3. Test order confirmation methods
echo "<h2>3. Order Confirmation Method Test</h2>";

if (class_exists('Loadlink_Order_Confirmation')) {
    $reflection = new ReflectionClass('Loadlink_Order_Confirmation');
    
    // Check confirm_freight_order method
    if ($reflection->hasMethod('confirm_freight_order')) {
        $method = $reflection->getMethod('confirm_freight_order');
        $parameters = $method->getParameters();
        
        echo "Loadlink_Order_Confirmation::confirm_freight_order parameters:<br>";
        foreach ($parameters as $param) {
            echo "- " . $param->getName() . " (" . ($param->getType() ? $param->getType() : 'mixed') . ")<br>";
        }
        
        if (count($parameters) === 3) {
            echo "✅ Order confirmation method signature updated correctly (3 parameters)<br>";
        } else {
            echo "❌ Order confirmation method has " . count($parameters) . " parameters (expected 3)<br>";
        }
    } else {
        echo "❌ confirm_freight_order method not found<br>";
    }
}

echo "<h2>Test Complete</h2>";
echo "<p>The payload format has been updated to match the reference format without freight_id.</p>";
echo "<p>Key changes:</p>";
echo "<ul>";
echo "<li>✅ Removed freight_id from payload</li>";
echo "<li>✅ Updated method signatures</li>";
echo "<li>✅ Updated method calls</li>";
echo "<li>✅ Updated logging and validation</li>";
echo "</ul>";
?>

