<?php
/**
 * Test script for WooCommerce Blocks checkout compatibility
 * Place this file in your WordPress root directory and access via browser
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    // Load WordPress
    require_once('wp-config.php');
    require_once('wp-load.php');
}

// Check if WooCommerce is active
if (!class_exists('WooCommerce')) {
    die('WooCommerce is not active');
}

echo "<h1>Loadlink Blocks Checkout Compatibility Test</h1>";

// 1. Check Blocks checkout availability
echo "<h2>1. WooCommerce Blocks Checkout Status</h2>";

// Check if Blocks are available
if (class_exists('Automattic\WooCommerce\Blocks\Package')) {
    echo "✅ WooCommerce Blocks package is available<br>";
    
    // Check Blocks version
    if (defined('Automattic\WooCommerce\Blocks\Package::VERSION')) {
        echo "✅ Blocks version: " . \Automattic\WooCommerce\Blocks\Package::VERSION . "<br>";
    }
} else {
    echo "❌ WooCommerce Blocks package not found<br>";
}

// Check if checkout page uses blocks
$checkout_page_id = wc_get_page_id('checkout');
if ($checkout_page_id) {
    $checkout_page = get_post($checkout_page_id);
    if ($checkout_page && has_blocks($checkout_page->post_content)) {
        echo "✅ Checkout page uses blocks<br>";
        
        // Check for specific checkout blocks
        if (strpos($checkout_page->post_content, 'wp:woocommerce/checkout') !== false) {
            echo "✅ WooCommerce checkout block found<br>";
        } else {
            echo "⚠️ WooCommerce checkout block not found<br>";
        }
    } else {
        echo "❌ Checkout page does not use blocks<br>";
    }
} else {
    echo "❌ Checkout page not found<br>";
}

// 2. Test hook registration
echo "<h2>2. Blocks Checkout Hook Registration</h2>";

global $wp_filter;

$blocks_hooks = [
    'woocommerce_store_api_checkout_order_processed',
    'woocommerce_store_api_checkout_update_order_meta',
    'woocommerce_new_order',
    'woocommerce_payment_complete'
];

foreach ($blocks_hooks as $hook) {
    if (isset($wp_filter[$hook])) {
        echo "✅ Hook '$hook' is registered<br>";
        
        // Check for Loadlink callbacks
        $callbacks = $wp_filter[$hook]->callbacks;
        $found_loadlink = false;
        
        foreach ($callbacks as $priority => $hooks) {
            foreach ($hooks as $callback) {
                if (is_array($callback['function']) && 
                    is_string($callback['function'][0]) && 
                    strpos($callback['function'][0], 'Loadlink') !== false) {
                    echo "&nbsp;&nbsp;✅ Found Loadlink callback: " . $callback['function'][0] . "::" . $callback['function'][1] . " (priority: $priority)<br>";
                    $found_loadlink = true;
                }
            }
        }
        
        if (!$found_loadlink) {
            echo "&nbsp;&nbsp;❌ No Loadlink callbacks found for '$hook'<br>";
        }
    } else {
        echo "❌ Hook '$hook' is NOT registered<br>";
    }
}

// 3. Test AJAX endpoints
echo "<h2>3. AJAX Endpoints</h2>";

$ajax_actions = [
    'loadlink_confirm_order',
    'loadlink_set_dropoff_type'
];

foreach ($ajax_actions as $action) {
    if (has_action("wp_ajax_$action") || has_action("wp_ajax_nopriv_$action")) {
        echo "✅ AJAX action '$action' is registered<br>";
    } else {
        echo "❌ AJAX action '$action' is NOT registered<br>";
    }
}

// 4. Test JavaScript assets
echo "<h2>4. JavaScript Assets</h2>";

$js_file = plugin_dir_path(__FILE__) . 'wp-content/plugins/loadlink-woocommerce/assets/loadlink-checkout.js';
if (file_exists($js_file)) {
    echo "✅ JavaScript file exists<br>";
    
    $js_content = file_get_contents($js_file);
    
    // Check for Blocks-specific functionality
    $blocks_features = [
        'handleBlocksCheckoutConfirmation',
        'triggerBlocksOrderConfirmation',
        'initBlocksCheckout',
        'checkout_success',
        'checkout_error'
    ];
    
    foreach ($blocks_features as $feature) {
        if (strpos($js_content, $feature) !== false) {
            echo "✅ JavaScript feature '$feature' found<br>";
        } else {
            echo "❌ JavaScript feature '$feature' NOT found<br>";
        }
    }
} else {
    echo "❌ JavaScript file not found<br>";
}

// 5. Test session handling
echo "<h2>5. Session Handling</h2>";

if (function_exists('WC') && WC()->session) {
    echo "✅ WooCommerce session is available<br>";
    
    // Test session data storage
    WC()->session->set('loadlink_test_data', ['test' => 'value']);
    $test_data = WC()->session->get('loadlink_test_data');
    
    if ($test_data && $test_data['test'] === 'value') {
        echo "✅ Session data storage/retrieval works<br>";
    } else {
        echo "❌ Session data storage/retrieval failed<br>";
    }
    
    // Clean up test data
    WC()->session->set('loadlink_test_data', null);
} else {
    echo "❌ WooCommerce session not available<br>";
}

// 6. Test order meta handling
echo "<h2>6. Order Meta Handling</h2>";

// Create a test order
$test_order = wc_create_order();
if ($test_order) {
    echo "✅ Test order created: #" . $test_order->get_id() . "<br>";
    
    // Test meta data storage
    $test_order->update_meta_data('_loadlink_test_meta', 'test_value');
    $test_order->update_meta_data('_loadlink_blocks_checkout', 'yes');
    $test_order->save();
    
    // Test meta data retrieval
    $test_meta = $test_order->get_meta('_loadlink_test_meta');
    $blocks_meta = $test_order->get_meta('_loadlink_blocks_checkout');
    
    if ($test_meta === 'test_value' && $blocks_meta === 'yes') {
        echo "✅ Order meta data storage/retrieval works<br>";
    } else {
        echo "❌ Order meta data storage/retrieval failed<br>";
    }
    
    // Clean up test order
    $test_order->delete(true);
    echo "✅ Test order cleaned up<br>";
} else {
    echo "❌ Could not create test order<br>";
}

// 7. Test Loadlink_Client availability
echo "<h2>7. Loadlink Client</h2>";

if (class_exists('Loadlink_Client')) {
    echo "✅ Loadlink_Client class exists<br>";
    
    // Test static method
    if (method_exists('Loadlink_Client', 'get_all_freight_data')) {
        echo "✅ get_all_freight_data method exists<br>";
    } else {
        echo "❌ get_all_freight_data method NOT found<br>";
    }
} else {
    echo "❌ Loadlink_Client class NOT found<br>";
}

// 8. Test order confirmation methods
echo "<h2>8. Order Confirmation Methods</h2>";

if (class_exists('Loadlink_Order_Confirmation')) {
    echo "✅ Loadlink_Order_Confirmation class exists<br>";
    
    $blocks_methods = [
        'handle_blocks_checkout_order',
        'save_freight_data_blocks',
        'handle_new_order',
        'handle_payment_complete',
        'ajax_confirm_order'
    ];
    
    foreach ($blocks_methods as $method) {
        if (method_exists('Loadlink_Order_Confirmation', $method)) {
            echo "✅ Method '$method' exists<br>";
        } else {
            echo "❌ Method '$method' NOT found<br>";
        }
    }
} else {
    echo "❌ Loadlink_Order_Confirmation class NOT found<br>";
}

// 9. Test HPOS compatibility
echo "<h2>9. HPOS Compatibility</h2>";

if (class_exists('Automattic\WooCommerce\Utilities\OrderUtil')) {
    if (\Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled()) {
        echo "⚠️ HPOS is enabled - this may affect order processing timing<br>";
        
        // Test HPOS order creation
        try {
            $hpos_order = wc_create_order();
            if ($hpos_order) {
                echo "✅ HPOS order creation works<br>";
                $hpos_order->delete(true);
            } else {
                echo "❌ HPOS order creation failed<br>";
            }
        } catch (Exception $e) {
            echo "❌ HPOS order creation error: " . $e->getMessage() . "<br>";
        }
    } else {
        echo "✅ HPOS is not enabled<br>";
    }
} else {
    echo "⚠️ OrderUtil class not found - cannot check HPOS status<br>";
}

// 10. Recommendations
echo "<h2>10. Recommendations</h2>";

echo "<h3>For Traditional Checkout:</h3>";
echo "<ul>";
echo "<li>Use <code>woocommerce_checkout_order_processed</code> hook</li>";
echo "<li>Session data should be available during checkout</li>";
echo "<li>Order confirmation happens immediately after order creation</li>";
echo "</ul>";

echo "<h3>For Blocks Checkout:</h3>";
echo "<ul>";
echo "<li>Use <code>woocommerce_store_api_checkout_order_processed</code> hook</li>";
echo "<li>Session data might not be available - use API fallback</li>";
echo "<li>Order confirmation might need to be triggered via AJAX</li>";
echo "<li>Consider using <code>woocommerce_new_order</code> and <code>woocommerce_payment_complete</code> hooks</li>";
echo "</ul>";

echo "<h3>Testing Steps:</h3>";
echo "<ol>";
echo "<li>Test with traditional checkout (disable blocks)</li>";
echo "<li>Test with Blocks checkout enabled</li>";
echo "<li>Test with HPOS enabled/disabled</li>";
echo "<li>Monitor WooCommerce logs during testing</li>";
echo "<li>Check browser console for JavaScript errors</li>";
echo "</ol>";

echo "<h2>Test Complete</h2>";
echo "<p>Review the results above and address any issues found. The system should work with both traditional and Blocks checkout.</p>";
?>

