<?php
class Loadlink_Settings {
    public static function init() {
        add_action('admin_menu', [__CLASS__, 'add_admin_menu'], 10); // Priority 10 to run after main menu
        add_action('admin_init', [__CLASS__, 'register_settings']);
        add_action('wp_ajax_test_loadlink_api', [__CLASS__, 'test_api_ajax']);
        add_action('wp_ajax_clear_loadlink_cache', [__CLASS__, 'clear_cache_ajax']);
        add_action('wp_ajax_get_loadlink_freight_data', [__CLASS__, 'get_freight_data_ajax']);
        add_action('wp_ajax_loadlink_quick_edit_product', [__CLASS__, 'quick_edit_product_ajax']);
        add_action('admin_notices', [__CLASS__, 'admin_notices']);

		// Inject Store Phone fields into WooCommerce → Settings → General (Store Address)
		add_filter('woocommerce_general_settings', [__CLASS__, 'inject_store_phone_fields']);
    }
    
    public static function add_admin_menu() {
        // Settings submenu (main menu is handled by Loadlink_Admin_Dashboard)
        add_submenu_page(
            'loadlink-dashboard',
            'Settings',
            'Settings',
            'manage_woocommerce',
            'loadlink-settings',
            [__CLASS__, 'settings_page']
        );
        
        // About submenu
        add_submenu_page(
            'loadlink-dashboard',
            'About',
            'About',
            'manage_woocommerce',
            'loadlink-about',
            [__CLASS__, 'about_page']
        );
        
        // Instructions submenu
        add_submenu_page(
            'loadlink-dashboard',
            'Instructions',
            'Instructions',
            'manage_woocommerce',
            'loadlink-instructions',
            [__CLASS__, 'instructions_page']
        );
        
        // Notices submenu
        add_submenu_page(
            'loadlink-dashboard',
            'Notices',
            'Notices',
            'manage_woocommerce',
            'loadlink-notices',
            [__CLASS__, 'notices_page']
        );
    }
    
    public static function register_settings() {
        register_setting('loadlink_settings', 'loadlink_api_url');
        register_setting('loadlink_settings', 'loadlink_api_key');
        register_setting('loadlink_settings', 'loadlink_api_secret');
        register_setting('loadlink_settings', 'loadlink_user_id');
        register_setting('loadlink_settings', 'loadlink_pickup_building_type');
        register_setting('loadlink_settings', 'loadlink_delivery_building_type');
        register_setting('loadlink_settings', 'loadlink_packaging_code');
        register_setting('loadlink_settings', 'loadlink_enable_logging');
        register_setting('loadlink_settings', 'loadlink_mock_mode');
        register_setting('loadlink_settings', 'loadlink_disable_throttling');

		// Store phone option used for pickup payload (displayed for convenience in this page)
		register_setting('loadlink_settings', 'woocommerce_store_phone');
    }

	/**
	 * Add Store Phone field after the postcode in WooCommerce General settings.
	 */
	public static function inject_store_phone_fields($settings) {
		$new_settings = [];
		foreach ($settings as $setting) {
			$new_settings[] = $setting;
			if (isset($setting['id']) && $setting['id'] === 'woocommerce_store_postcode') {
				$new_settings[] = [
					'title'    => __('Store phone', 'loadlink'),
					'id'       => 'woocommerce_store_phone',
					'type'     => 'text',
					'css'      => 'min-width:300px;',
					'desc'     => __('Used for pickup contact details (also used by Loadlink).', 'loadlink'),
					'desc_tip' => true,
				];
			}
		}
		return $new_settings;
	}
    
    public static function settings_page() {
        ?>
        <div class="wrap">
            <h1 class="screen-reader-text">Settings</h1>
            <hr class="wp-header-end">
            <div class="loadlink-header">
                <div class="loadlink-logo">
                    <svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px"
                         viewBox="0 0 689 101.9" style="enable-background:new 0 0 689 101.9;" xml:space="preserve" width="200" height="30">
                    <style type="text/css">
                        .st0{fill:#1E3D5B;}
                        .st1{fill:#F37D48;}
                    </style>
                    <g>
                        <g>
                            <path class="st0" d="M193.6,77.5c-1.5-1-1.6-2.5-0.2-4.5l18.8-28.4c1.4-2,3.4-3.5,6.2-4.5c2.7-0.9,6.4-1.4,10.9-1.4l40,0.1
                                c4.6,0,7.6,0.5,9,1.4c1.4,1,1.4,2.5,0,4.5l-18.8,28.4c-1.4,2-3.4,3.5-6,4.5c-2.6,0.9-6.3,1.4-10.8,1.4l-40-0.1
                                C198.1,79,195.1,78.5,193.6,77.5z M223.1,69.2c0.6,0.4,1.8,0.6,3.7,0.6l3.5,0c2,0,3.5-0.2,4.6-0.6c1.1-0.4,2-1.2,2.8-2.4
                                l10.5-15.9c0.8-1.2,0.9-2,0.4-2.4c-0.5-0.4-1.8-0.6-3.7-0.6h-3.5c-1.9,0-3.4,0.2-4.6,0.6c-1.1,0.4-2.1,1.2-2.9,2.4l-10.5,15.9
                                C222.6,68,222.5,68.8,223.1,69.2z"/>
                            <polygon class="st1" points="458,69.9 478.6,38.8 452.7,38.8 426,79 470.1,79 481.1,69.9 		"/>
                            <path class="st0" d="M434.4,40.2c-1.4-1-4.3-1.5-9-1.5l-58.9,0l-11,9.2l11,0L345.9,79l52.9,0.1c4.5,0,8.1-0.5,10.8-1.4
                                c2.7-0.9,4.7-2.4,6.1-4.5l18.8-28.4C435.9,42.7,435.9,41.2,434.4,40.2z M404.5,51L394,66.9c-0.8,1.2-1.7,2-2.8,2.4
                                c-1.1,0.4-2.6,0.6-4.6,0.6l-8.6,0l14.6-22h8.6c2,0,3.2,0.3,3.8,0.7C405.4,49,405.3,49.8,404.5,51z"/>
                            <path class="st1" d="M563.6,79L552,64l-9.9,15l-25.9,0l26.6-40.2h18.7l11.6,15.1l10-15.1H609L582.4,79H563.6z"/>
                            <polygon class="st1" points="534.3,38.8 508.4,38.8 481.7,79 507.6,79 		"/>
                            <polygon class="st1" points="651.3,58.9 683.7,38.8 657.8,38.8 633.2,54.3 643.4,38.8 617.5,38.8 590.8,79 616.7,79 627,63.6 
                                631.1,79 656.9,79 		"/>
                            <polygon class="st0" points="343.8,38.8 307.8,38.8 261.6,79 300.4,79 311.4,69.9 297.3,69.9 317.9,51.8 314.6,79 339.9,79 		"/>
                            <polygon class="st0" points="165.7,69.9 186.2,38.8 160.3,38.8 133.7,79 177.8,79 188.8,69.9 		"/>
                        </g>
                        <g>
                            <g>
                                <path class="st1" d="M119.2,22.9c-6.6,1.8-13.5,4.1-20.7,6.9l6.9,1.4c5.5-2.6,11.1-5,16.7-7.2L119.2,22.9z"/>
                                <path class="st1" d="M45.4,58.2H62c6.5-5.2,13.5-10.1,20.9-14.7l-10.2-1.9C63.5,46.5,54.4,52,45.4,58.2z"/>
                                <path class="st1" d="M134.9,19.5c-3.4,0.5-6.9,1.2-10.6,2.1l2.1,0.8C129.3,21.3,132.1,20.4,134.9,19.5z"/>
                                <path class="st1" d="M77.8,39l9.7,1.7c4.4-2.6,8.9-5,13.5-7.3l-7.5-1.6C88.3,33.9,83.1,36.3,77.8,39z"/>
                            </g>
                            <g>
                                <path class="st1" d="M38.1,63.4c-0.5,0.4-1,0.7-1.5,1.1C25.3,73.2,14.3,82.9,4.4,93.7h24c4.9-7.3,17.3-20.9,27.3-30.3H38.1z"/>
                            </g>
                            <path class="st0" d="M64.2,93.7c14.6-24.7,35.9-47.8,69.9-67.8c-45.7,12.5-84,55.4-91.5,67.8H64.2z"/>
                            <g>
                                <path class="st0" d="M120.3,34.4c-3,2-6,4-8.8,6.1c1,3.2,1.5,6.5,1.5,10c0,18.9-15.3,34.1-34.1,34.1c-0.6,0-1.1,0-1.7,0
                                    c-2.4,3.1-4.6,6.2-6.7,9.5l-0.1,0.1c2.7,0.5,5.5,0.8,8.4,0.8c24.6,0,44.5-19.9,44.5-44.5C123.4,44.9,122.3,39.5,120.3,34.4z"/>
                            </g>
                            <g>
                                <g>
                                    <path class="st0" d="M78.9,6.1C54.3,6.1,34.4,26,34.4,50.6c0,0.7,0,1.3,0,2c3.5-2.2,7-4.2,10.6-6.2
                                        c2.1-16.9,16.5-29.9,33.9-29.9c4.1,0,8,0.7,11.6,2l15.3-3.3C98.4,9.5,89,6.1,78.9,6.1z"/>
                                </g>
                            </g>
                            <path class="st0" d="M118.3,18.9c-16.1,1.5-40.9,7.4-50,9.9c-8.9,2.5-16.2,5.1-20.9,6.8c-1.7,3.4-12.6,13-13,17
                                c9.6-5.7,25-13.2,35.8-17.8c9.2-4,27.1-10,30.3-10.9C103.8,22.9,118.3,18.9,118.3,18.9z"/>
                        </g>
                    </g>
                    </svg>
                </div>
                <h1>Settings</h1>
            </div>
            <form method="post" action="options.php">
                <?php
                settings_fields('loadlink_settings');
                do_settings_sections('loadlink_settings');
                ?>
                <table class="form-table">
                    <tr>
                        <th scope="row">API URL</th>
                        <td>
                            <input type="text" name="loadlink_api_url" value="<?php echo esc_attr(get_option('loadlink_api_url', 'https://parcelfreight.loadlink.net.au/api/v1/partner/freight/get_price')); ?>" class="regular-text" style="width: 100%;" />
                            <p class="description">The full API endpoint URL (default: https://parcelfreight.loadlink.net.au/api/v1/partner/freight/get_price)</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">API Key</th>
                        <td>
                            <input type="text" name="loadlink_api_key" value="<?php echo esc_attr(get_option('loadlink_api_key', '')); ?>" class="regular-text" />
                            <p class="description">Your Loadlink API public key</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">API Secret</th>
                        <td>
                            <input type="password" name="loadlink_api_secret" value="<?php echo esc_attr(get_option('loadlink_api_secret', '')); ?>" class="regular-text" />
                            <p class="description">Your Loadlink API secret key</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">User ID</th>
                        <td>
                            <input type="number" name="loadlink_user_id" value="<?php echo esc_attr(get_option('loadlink_user_id', 10)); ?>" class="regular-text" />
                            <p class="description">Your Loadlink account user ID (if still required)</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Pickup Building Type</th>
                        <td>
                            <select name="loadlink_pickup_building_type">
                                <option value="Residential" <?php selected(get_option('loadlink_pickup_building_type', 'Residential'), 'Residential'); ?>>Residential</option>
                                <option value="Commercial" <?php selected(get_option('loadlink_pickup_building_type', 'Residential'), 'Commercial'); ?>>Commercial</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Delivery Building Type</th>
                        <td>
                            <select name="loadlink_delivery_building_type">
                                <option value="Residential" <?php selected(get_option('loadlink_delivery_building_type', 'Residential'), 'Residential'); ?>>Residential</option>
                                <option value="Commercial" <?php selected(get_option('loadlink_delivery_building_type', 'Residential'), 'Commercial'); ?>>Commercial</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Store Phone</th>
                        <td>
                            <input type="text" name="woocommerce_store_phone" value="<?php echo esc_attr(get_option('woocommerce_store_phone', '')); ?>" class="regular-text" />
                            <p class="description">Phone number for pickups (used in pickup payload). Example: +61 2 9876 5432 or 0401 234 567</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">Packaging Code</th>
                        <td>
                            <input type="text" name="loadlink_packaging_code" value="<?php echo esc_attr(get_option('loadlink_packaging_code', 'BX')); ?>" class="regular-text" />
                            <p class="description">Default packaging code (BX = Box)</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Enable Logging</th>
                        <td>
                            <input type="checkbox" name="loadlink_enable_logging" value="1" <?php checked(get_option('loadlink_enable_logging', 0), 1); ?> />
                            <p class="description">Enable detailed logging for debugging</p>
                        </td>
                    </tr>
                <tr>
                    <th scope="row">Mock API Mode</th>
                    <td>
                        <input type="checkbox" name="loadlink_mock_mode" value="1" <?php checked(get_option('loadlink_mock_mode', 0), 1); ?> />
                        <p class="description">Return sample Loadlink rates when the real API is unavailable.</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Disable Throttling</th>
                    <td>
                        <input type="checkbox" name="loadlink_disable_throttling" value="1" <?php checked(get_option('loadlink_disable_throttling', 0), 1); ?> />
                        <p class="description">Disable API throttling for testing purposes. <strong>Warning:</strong> This may cause excessive API calls.</p>
                    </td>
                </tr>
                </table>
                <?php submit_button(); ?>
            </form>
            
            <h2>Test API Connection</h2>
            <p>Test the API connection with real data:</p>
            
            <table class="form-table">
                <tr>
                    <th scope="row">Test Destination</th>
                    <td>
                        <input type="text" id="test-destination-city" placeholder="City" class="regular-text" style="width: 30%;" />
                        <input type="text" id="test-destination-postcode" placeholder="Postcode" class="regular-text" style="width: 20%; margin-left: 5px;" />
                        <select id="test-destination-state" style="margin-left: 5px;">
                            <option value="">Select State</option>
                            <option value="NSW">NSW</option>
                            <option value="VIC">VIC</option>
                            <option value="QLD">QLD</option>
                            <option value="SA">SA</option>
                            <option value="WA">WA</option>
                            <option value="TAS">TAS</option>
                            <option value="NT">NT</option>
                            <option value="ACT">ACT</option>
                        </select>
                        <p class="description">Enter destination address for testing (e.g., MOOCKRA, 5432, SA)</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Test Package</th>
                    <td>
                        <input type="number" id="test-length" placeholder="Length (cm)" value="3" style="width: 15%;" />
                        <input type="number" id="test-width" placeholder="Width (cm)" value="3" style="width: 15%; margin-left: 5px;" />
                        <input type="number" id="test-height" placeholder="Height (cm)" value="3" style="width: 15%; margin-left: 5px;" />
                        <input type="number" id="test-weight" placeholder="Weight (kg)" value="9" step="0.1" style="width: 15%; margin-left: 5px;" />
                        <p class="description">Package dimensions and weight for testing</p>
                    </td>
                </tr>
            </table>
            
            <button type="button" id="test-loadlink-api" class="button">Test API with Real Data</button>
            <button type="button" id="test-ajax-endpoint" class="button" style="margin-left: 10px;">Test AJAX Endpoint</button>
            <button type="button" id="test-classes" class="button" style="margin-left: 10px;">Test Class Loading</button>
            <button type="button" id="clear-cache" class="button" style="margin-left: 10px;">Clear Cache</button>
            <button type="button" id="view-freight-data" class="button" style="margin-left: 10px;">View Freight Data</button>
            <div id="api-test-result" style="margin-top: 10px;"></div>
            
            <script>
            document.getElementById('test-loadlink-api').addEventListener('click', function() {
                var resultDiv = document.getElementById('api-test-result');
                resultDiv.innerHTML = 'Testing API connection...';
                
                // Get form data
                var city = document.getElementById('test-destination-city').value;
                var postcode = document.getElementById('test-destination-postcode').value;
                var state = document.getElementById('test-destination-state').value;
                var length = document.getElementById('test-length').value;
                var width = document.getElementById('test-width').value;
                var height = document.getElementById('test-height').value;
                var weight = document.getElementById('test-weight').value;
                
                // Validate required fields
                if (!city || !postcode || !state) {
                    resultDiv.innerHTML = '<div style="color: red;">Please fill in all destination fields (City, Postcode, State)</div>';
                    return;
                }
                
                var formData = new FormData();
                formData.append('action', 'test_loadlink_api');
                formData.append('nonce', '<?php echo wp_create_nonce('test_loadlink_api'); ?>');
                formData.append('test_city', city);
                formData.append('test_postcode', postcode);
                formData.append('test_state', state);
                formData.append('test_length', length);
                formData.append('test_width', width);
                formData.append('test_height', height);
                formData.append('test_weight', weight);
                
                fetch(ajaxurl, {
                    method: 'POST',
                    body: formData
                })
                .then(response => {
                    if (!response.ok) {
                        throw new Error('HTTP error! status: ' + response.status);
                    }
                    return response.text().then(text => {
                        try {
                            return JSON.parse(text);
                        } catch (e) {
                            console.error('Response text:', text);
                            throw new Error('Invalid JSON response: ' + e.message + '. Response: ' + text.substring(0, 200));
                        }
                    });
                })
                .then(data => {
                    var html = '<h3>API Test Results:</h3>';
                    if (data.success) {
                        html += '<div style="color: green; margin-bottom: 10px;">✅ ' + data.data.message + '</div>';
                        html += '<h4>Raw API Response:</h4>';
                        html += '<pre style="background: #f1f1f1; padding: 10px; border-radius: 4px; overflow-x: auto;">' + JSON.stringify(data.data.api_response, null, 2) + '</pre>';
                        html += '<h4>Parsed WooCommerce Rates:</h4>';
                        html += '<pre style="background: #f1f1f1; padding: 10px; border-radius: 4px; overflow-x: auto;">' + JSON.stringify(data.data.rates, null, 2) + '</pre>';
                    } else {
                        html += '<div style="color: red;">❌ ' + data.data.message + '</div>';
                        if (data.data.api_response) {
                            html += '<h4>API Response:</h4>';
                            html += '<pre style="background: #f1f1f1; padding: 10px; border-radius: 4px; overflow-x: auto;">' + JSON.stringify(data.data.api_response, null, 2) + '</pre>';
                        }
                    }
                    resultDiv.innerHTML = html;
                })
                .catch(error => {
                    resultDiv.innerHTML = '<div style="color: red;">Error: ' + error.message + '</div>';
                });
            });
            
            // Simple AJAX endpoint test
            document.getElementById('test-ajax-endpoint').addEventListener('click', function() {
                var resultDiv = document.getElementById('api-test-result');
                resultDiv.innerHTML = 'Testing AJAX endpoint...';
                
                var formData = new FormData();
                formData.append('action', 'test_loadlink_api');
                formData.append('nonce', '<?php echo wp_create_nonce('test_loadlink_api'); ?>');
                formData.append('test_simple', '1');
                
                fetch(ajaxurl, {
                    method: 'POST',
                    body: formData
                })
                .then(response => {
                    if (!response.ok) {
                        throw new Error('HTTP error! status: ' + response.status);
                    }
                    return response.text().then(text => {
                        try {
                            return JSON.parse(text);
                        } catch (e) {
                            console.error('Response text:', text);
                            throw new Error('Invalid JSON response: ' + e.message + '. Response: ' + text.substring(0, 200));
                        }
                    });
                })
                .then(data => {
                    resultDiv.innerHTML = '<div style="color: green;">✅ AJAX Endpoint Working: ' + JSON.stringify(data) + '</div>';
                })
                .catch(error => {
                    resultDiv.innerHTML = '<div style="color: red;">❌ AJAX Error: ' + error.message + '</div>';
                });
            });
            
            // Test class loading
            document.getElementById('test-classes').addEventListener('click', function() {
                var resultDiv = document.getElementById('api-test-result');
                resultDiv.innerHTML = 'Testing class loading...';
                
                var formData = new FormData();
                formData.append('action', 'test_loadlink_api');
                formData.append('nonce', '<?php echo wp_create_nonce('test_loadlink_api'); ?>');
                formData.append('test_classes', '1');
                
                fetch(ajaxurl, {
                    method: 'POST',
                    body: formData
                })
                .then(response => {
                    if (!response.ok) {
                        throw new Error('HTTP error! status: ' + response.status);
                    }
                    return response.text().then(text => {
                        try {
                            return JSON.parse(text);
                        } catch (e) {
                            console.error('Response text:', text);
                            throw new Error('Invalid JSON response: ' + e.message + '. Response: ' + text.substring(0, 200));
                        }
                    });
                })
                .then(data => {
                    var html = '<div style="color: green;">✅ Class Loading Test Results:</div>';
                    html += '<pre style="background: #f1f1f1; padding: 10px; border-radius: 4px; overflow-x: auto;">' + JSON.stringify(data.data.results, null, 2) + '</pre>';
                    resultDiv.innerHTML = html;
                })
                .catch(error => {
                    resultDiv.innerHTML = '<div style="color: red;">❌ Class Loading Error: ' + error.message + '</div>';
                });
            });
            
            // Clear cache button
            document.getElementById('clear-cache').addEventListener('click', function() {
                var resultDiv = document.getElementById('api-test-result');
                resultDiv.innerHTML = 'Clearing cache...';
                
                var formData = new FormData();
                formData.append('action', 'clear_loadlink_cache');
                formData.append('nonce', '<?php echo wp_create_nonce('clear_loadlink_cache'); ?>');
                
                fetch(ajaxurl, {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        resultDiv.innerHTML = '<div style="color: green;">✅ Cache cleared successfully</div>';
                    } else {
                        resultDiv.innerHTML = '<div style="color: red;">❌ Error clearing cache: ' + data.data.message + '</div>';
                    }
                })
                .catch(error => {
                    resultDiv.innerHTML = '<div style="color: red;">❌ Error: ' + error.message + '</div>';
                });
            });
            
            // View freight data button
            document.getElementById('view-freight-data').addEventListener('click', function() {
                var resultDiv = document.getElementById('api-test-result');
                resultDiv.innerHTML = 'Loading freight data...';
                
                var formData = new FormData();
                formData.append('action', 'get_loadlink_freight_data');
                formData.append('nonce', '<?php echo wp_create_nonce('get_loadlink_freight_data'); ?>');
                
                fetch(ajaxurl, {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        var html = '<div style="color: green;">✅ Freight Data Retrieved:</div>';
                        html += '<h4>Freight ID:</h4>';
                        html += '<pre style="background: #f1f1f1; padding: 10px; border-radius: 4px; overflow-x: auto;">' + (data.data.freight_id || 'Not available') + '</pre>';
                        html += '<h4>Service Quote IDs:</h4>';
                        html += '<pre style="background: #f1f1f1; padding: 10px; border-radius: 4px; overflow-x: auto;">' + JSON.stringify(data.data.service_quote_ids || {}, null, 2) + '</pre>';
                        html += '<h4>Full Freight Data:</h4>';
                        html += '<pre style="background: #f1f1f1; padding: 10px; border-radius: 4px; overflow-x: auto;">' + JSON.stringify(data.data, null, 2) + '</pre>';
                        resultDiv.innerHTML = html;
                    } else {
                        resultDiv.innerHTML = '<div style="color: red;">❌ Error: ' + data.data.message + '</div>';
                    }
                })
                .catch(error => {
                    resultDiv.innerHTML = '<div style="color: red;">❌ Error: ' + error.message + '</div>';
                });
            });
            </script>
            

        </div>
        <?php
    }
    
    public static function test_api_ajax() {
        // Set proper headers for JSON response
        header('Content-Type: application/json');
        
        // Enable error reporting for debugging
        error_reporting(E_ALL);
        ini_set('display_errors', 1);
        
        try {
            check_ajax_referer('test_loadlink_api', 'nonce');
            
            if (!current_user_can('manage_options')) {
                wp_send_json_error(['message' => 'Unauthorized']);
                return;
            }
        } catch (Exception $e) {
            wp_send_json_error(['message' => 'Nonce verification failed: ' . $e->getMessage()]);
            return;
        }
        
        // Ensure required classes are loaded when called via admin-ajax
        if (!class_exists('Loadlink_Client')) {
            $plugin_path = defined('LOADLINK_PLUGIN_PATH') ? LOADLINK_PLUGIN_PATH : plugin_dir_path(dirname(__FILE__));
            require_once $plugin_path . 'includes/class-loadlink-client.php';
        }
        if (!class_exists('Loadlink_Logger')) {
            $plugin_path = defined('LOADLINK_PLUGIN_PATH') ? LOADLINK_PLUGIN_PATH : plugin_dir_path(dirname(__FILE__));
            require_once $plugin_path . 'includes/class-loadlink-logger.php';
        }

        // Debug: Log the incoming request
        error_log('Loadlink AJAX Request - POST data: ' . print_r($_POST, true));
        
        // Simple test response first
        if (isset($_POST['test_simple'])) {
            wp_send_json_success(['message' => 'AJAX endpoint is working!', 'timestamp' => current_time('mysql')]);
            return;
        }
        
        // Test if classes can be loaded
        if (isset($_POST['test_classes'])) {
            $plugin_path = defined('LOADLINK_PLUGIN_PATH') ? LOADLINK_PLUGIN_PATH : plugin_dir_path(dirname(__FILE__));
            $client_file = $plugin_path . 'includes/class-loadlink-client.php';
            $logger_file = $plugin_path . 'includes/class-loadlink-logger.php';
            
            $results = [
                'plugin_path' => $plugin_path,
                'client_file_exists' => file_exists($client_file),
                'logger_file_exists' => file_exists($logger_file),
                'client_class_exists' => class_exists('Loadlink_Client'),
                'logger_class_exists' => class_exists('Loadlink_Logger')
            ];
            
            wp_send_json_success(['message' => 'Class loading test', 'results' => $results]);
            return;
        }
        
        // Get test data from form
        $test_city = sanitize_text_field($_POST['test_city'] ?? '');
        $test_postcode = sanitize_text_field($_POST['test_postcode'] ?? '');
        $test_state = sanitize_text_field($_POST['test_state'] ?? '');
        $test_length = floatval($_POST['test_length'] ?? 3);
        $test_width = floatval($_POST['test_width'] ?? 3);
        $test_height = floatval($_POST['test_height'] ?? 3);
        $test_weight = floatval($_POST['test_weight'] ?? 9);
        
        // Validate required fields
        if (empty($test_city) || empty($test_postcode) || empty($test_state)) {
            wp_send_json_error([
                'message' => 'Please fill in all destination fields (City, Postcode, State)'
            ]);
        }
        
        // Get WooCommerce store address
        $store_city = get_option('woocommerce_store_city', '');
        $store_postcode = get_option('woocommerce_store_postcode', '');
        $store_country = get_option('woocommerce_default_country', 'AU');
        $base_location = function_exists('wc_get_base_location') ? wc_get_base_location() : ['country' => $store_country, 'state' => ''];
        $store_state = isset($base_location['state']) ? $base_location['state'] : '';
        
        // Create test package data with real WooCommerce store address
        $test_package = [
            'destination' => [
                'address' => '123 Test Street',
                'city' => $test_city,
                'postcode' => $test_postcode,
                'state' => $test_state,
                'country' => 'AU'
            ],
            'contents' => [
                [
                    // Anonymous class to mimic WC_Product getters used by the client
                    'data' => new class($test_length, $test_width, $test_height, $test_weight) {
                        private $length, $width, $height, $weight;
                        public function __construct($l, $w, $h, $wt) {
                            $this->length = $l;
                            $this->width = $w;
                            $this->height = $h;
                            $this->weight = $wt;
                        }
                        public function get_length() { return $this->length; }
                        public function get_width() { return $this->width; }
                        public function get_height() { return $this->height; }
                        public function get_weight() { return $this->weight; }
                        public function get_name() { return 'Test Product'; }
                    },
                    'quantity' => 1
                ]
            ]
        ];
        
        try {
            // Check if credentials are configured
            $api_key = get_option('loadlink_api_key', '');
            $api_secret = get_option('loadlink_api_secret', '');
            
            if (empty($api_key) || empty($api_secret)) {
                wp_send_json_error([
                    'message' => 'API credentials not configured. Please enter your API Key and API Secret.'
                ]);
            }
            
            try {
                $client = new Loadlink_Client();
            } catch (Exception $e) {
                wp_send_json_error([
                    'message' => 'Failed to create Loadlink_Client: ' . $e->getMessage()
                ]);
                return;
            }
            
            // Get the raw API response for debugging
            try {
                $api_payload = $client->build_api_payload($test_package);
            } catch (Exception $e) {
                wp_send_json_error([
                    'message' => 'Failed to build API payload: ' . $e->getMessage()
                ]);
                return;
            }
            
            try {
                $api_response = $client->make_api_request($api_payload);
            } catch (Exception $e) {
                wp_send_json_error([
                    'message' => 'Failed to make API request: ' . $e->getMessage()
                ]);
                return;
            }
            
            if (is_wp_error($api_response)) {
                wp_send_json_error([
                    'message' => 'API request failed: ' . $api_response->get_error_message(),
                    'api_response' => $api_response->get_error_message()
                ]);
            }
            
            try {
                $rates = $client->parse_api_response($api_response);
            } catch (Exception $e) {
                wp_send_json_error([
                    'message' => 'Failed to parse API response: ' . $e->getMessage()
                ]);
                return;
            }
            
            wp_send_json_success([
                'message' => 'API test completed successfully',
                'rates' => $rates,
                'api_response' => json_decode($api_response, true),
                'test_package' => $test_package,
                'store_address' => [
                    'city' => $store_city,
                    'postcode' => $store_postcode,
                    'state' => $store_state,
                    'country' => $store_country
                ],
                'credentials_configured' => true
            ]);
            
        } catch (Exception $e) {
            wp_send_json_error([
                'message' => 'API test failed: ' . $e->getMessage()
            ]);
        }
    }
    
    public static function clear_cache_ajax() {
        // Set proper headers for JSON response
        header('Content-Type: application/json');
        
        try {
            check_ajax_referer('clear_loadlink_cache', 'nonce');
            
            if (!current_user_can('manage_options')) {
                wp_send_json_error(['message' => 'Unauthorized']);
                return;
            }
            
            // Clear the cache
            if (class_exists('Loadlink_Client')) {
                Loadlink_Client::clear_cache();
                wp_send_json_success(['message' => 'Cache cleared successfully']);
            } else {
                wp_send_json_error(['message' => 'Loadlink_Client class not found']);
            }
            
        } catch (Exception $e) {
            wp_send_json_error(['message' => 'Cache clear failed: ' . $e->getMessage()]);
        }
    }
    
    public static function get_freight_data_ajax() {
        // Set proper headers for JSON response
        header('Content-Type: application/json');
        
        try {
            check_ajax_referer('get_loadlink_freight_data', 'nonce');
            
            if (!current_user_can('manage_options')) {
                wp_send_json_error(['message' => 'Unauthorized']);
                return;
            }
            
            // Get the freight data
            if (class_exists('Loadlink_Client')) {
                $freight_data = Loadlink_Client::get_all_freight_data();
                wp_send_json_success($freight_data);
            } else {
                wp_send_json_error(['message' => 'Loadlink_Client class not found']);
            }
            
        } catch (Exception $e) {
            wp_send_json_error(['message' => 'Failed to get freight data: ' . $e->getMessage()]);
        }
    }
    
    public static function dashboard_page() {
        // Calculate status and stats
        $api_key = get_option('loadlink_api_key');
        $api_secret = get_option('loadlink_api_secret');
        $is_configured = !empty($api_key) && !empty($api_secret);
        
        $freight_count = 0;
        // Count orders with Loadlink freight ID
        global $wpdb;
        // Check if table exists first to avoid errors
        if ($wpdb->get_var("SHOW TABLES LIKE '{$wpdb->postmeta}'") === $wpdb->postmeta) {
            $count_result = $wpdb->get_var("
                SELECT COUNT(post_id) 
                FROM {$wpdb->postmeta} 
                WHERE meta_key = '_loadlink_freight_id'
            ");
            $freight_count = $count_result ? (int)$count_result : 0;
        }
        ?>
        <div class="wrap loadlink-dashboard-wrapper">
            <!-- Hidden header to ensure WP admin notices appear above our custom banner -->
            <h1 class="screen-reader-text">Loadlink Parcel Freight</h1>
            <hr class="wp-header-end">

            <style>
                :root {
                    --ll-blue: #132B43;
                    --ll-blue-light: #1a3a5c;
                    --ll-orange: #FF7D44;
                    --ll-orange-light: #ff9966;
                    --ll-orange-dark: #e86a2e;
                    --ll-white: #ffffff;
                    --ll-gray-50: #f8f9fa;
                    --ll-gray-100: #f1f3f5;
                    --ll-gray-200: #e9ecef;
                    --ll-gray-300: #dee2e6;
                    --ll-gray-600: #6d7175;
                    --ll-success: #12b76a;
                    --ll-error: #f04438;
                    --ll-shadow-sm: 0 1px 2px rgba(19, 43, 67, 0.06);
                    --ll-shadow-lg: 0 4px 12px rgba(19, 43, 67, 0.1);
                    --ll-shadow-xl: 0 16px 48px rgba(19, 43, 67, 0.2);
                }
                
                .loadlink-dashboard-wrapper {
                    font-family: -apple-system, BlinkMacSystemFont, "San Francisco", "Segoe UI", Roboto, "Helvetica Neue", sans-serif;
                    max-width: 1200px;
                    margin: 20px auto;
                }
                
                .loadlink-hero {
                    background: linear-gradient(135deg, var(--ll-blue) 0%, var(--ll-blue-light) 50%, var(--ll-blue) 100%);
                    padding: 48px 40px;
                    border-radius: 16px;
                    color: var(--ll-white);
                    margin-bottom: 32px;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                    box-shadow: var(--ll-shadow-xl);
                    position: relative;
                    overflow: hidden;
                }
                
                .loadlink-hero::before {
                    content: "";
                    position: absolute;
                    bottom: 0;
                    left: 0;
                    right: 0;
                    height: 4px;
                    background: linear-gradient(90deg, var(--ll-orange) 0%, var(--ll-orange-light) 50%, var(--ll-orange) 100%);
                }

                .loadlink-hero::after {
                    content: "";
                    position: absolute;
                    top: 0;
                    right: 0;
                    bottom: 0;
                    left: 0;
                    background: radial-gradient(circle at 80% 20%, rgba(255, 125, 68, 0.1) 0%, transparent 40%),
                                radial-gradient(circle at 20% 80%, rgba(255, 255, 255, 0.05) 0%, transparent 30%);
                    pointer-events: none;
                }
                
                .hero-content {
                    z-index: 1;
                    position: relative;
                }

                .hero-content h1 {
                    font-size: 32px;
                    font-weight: 700;
                    margin: 0 0 12px 0;
                    color: white;
                    letter-spacing: -0.02em;
                }
                
                .hero-subtitle {
                    font-size: 16px;
                    opacity: 0.9;
                    margin: 0;
                    max-width: 400px;
                    line-height: 1.5;
                    font-weight: 400;
                }
                
                .hero-logo-container {
                    background: var(--ll-white);
                    padding: 16px 20px;
                    border-radius: 12px;
                    box-shadow: var(--ll-shadow-lg);
                    z-index: 1;
                    border: 1px solid rgba(255, 255, 255, 0.2);
                }

                .hero-logo-container svg {
                    height: 48px;
                    width: auto;
                    display: block;
                }
                
                .ll-grid {
                    display: grid;
                    grid-template-columns: 1fr 1fr;
                    gap: 24px;
                    margin-bottom: 24px;
                }
                
                .ll-card {
                    background: var(--ll-white);
                    border-radius: 12px;
                    padding: 24px;
                    box-shadow: var(--ll-shadow-sm);
                    height: 100%;
                    box-sizing: border-box;
                    transition: all 0.3s cubic-bezier(0.25, 0.8, 0.25, 1);
                    position: relative;
                }

                .ll-card:hover {
                    transform: translateY(-4px);
                    box-shadow: var(--ll-shadow-lg);
                }

                .ll-card::after {
                    content: "";
                    position: absolute;
                    bottom: 0;
                    left: 50%;
                    transform: translateX(-50%);
                    width: 0;
                    height: 3px;
                    background: linear-gradient(90deg, var(--ll-orange) 0%, var(--ll-orange-light) 100%);
                    border-radius: 3px 3px 0 0;
                    transition: width 0.3s ease;
                }

                .ll-card:hover::after {
                    width: 60%;
                }
                
                .ll-card-header {
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                    margin-bottom: 24px;
                }
                
                .ll-card-title {
                    font-size: 16px;
                    font-weight: 600;
                    margin: 0;
                    color: var(--ll-blue);
                }
                
                .ll-card-subtitle {
                    font-size: 13px;
                    color: var(--ll-gray-600);
                }
                
                .stat-row {
                    background: linear-gradient(135deg, var(--ll-gray-50) 0%, var(--ll-gray-100) 100%);
                    border-radius: 10px;
                    border: 1px solid var(--ll-gray-200);
                    padding: 16px;
                    margin-bottom: 16px;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                }
                
                .stat-label {
                    font-weight: 500;
                    color: var(--ll-blue);
                    font-size: 14px;
                }
                
                .status-badge {
                    display: flex;
                    align-items: center;
                    gap: 8px;
                    font-weight: 600;
                    font-size: 14px;
                }
                
                .status-dot {
                    width: 10px;
                    height: 10px;
                    border-radius: 50%;
                }
                
                .status-active { color: var(--ll-success); }
                .dot-active { background: var(--ll-success); box-shadow: 0 0 8px rgba(18, 183, 106, 0.4); animation: pulse 2s infinite; }
                
                .status-pending { color: var(--ll-orange); }
                .dot-pending { background: var(--ll-orange); }
                
                .stat-value {
                    font-size: 20px;
                    font-weight: 700;
                    color: var(--ll-blue);
                }
                
                .ll-btn {
                    display: block;
                    width: 100%;
                    padding: 12px;
                    text-align: center;
                    border-radius: 8px;
                    text-decoration: none;
                    font-weight: 500;
                    margin-bottom: 12px;
                    box-sizing: border-box;
                    transition: all 0.2s;
                    cursor: pointer;
                    font-size: 14px;
                }
                
                .ll-btn-primary {
                    background: linear-gradient(135deg, var(--ll-orange) 0%, var(--ll-orange-dark) 100%);
                    color: white !important;
                    border: none;
                    box-shadow: 0 2px 8px rgba(255, 125, 68, 0.3);
                }
                
                .ll-btn-primary:hover {
                    background: linear-gradient(135deg, var(--ll-orange-light) 0%, var(--ll-orange) 100%);
                    transform: translateY(-1px);
                    box-shadow: 0 4px 16px rgba(255, 125, 68, 0.4);
                }
                
                .ll-btn-secondary {
                    background: white;
                    border: 1px solid var(--ll-gray-300);
                    color: var(--ll-blue) !important;
                }
                
                .ll-btn-secondary:hover {
                    background: var(--ll-gray-50);
                    border-color: var(--ll-blue);
                }
                
                .guide-grid {
                    display: grid;
                    grid-template-columns: repeat(3, 1fr);
                    gap: 20px;
                }
                
                .guide-item {
                    background: var(--ll-gray-50);
                    padding: 20px;
                    border-radius: 8px;
                    transition: all 0.2s ease;
                }
                
                .guide-item:hover {
                     background: var(--ll-gray-100);
                }
                
                .guide-title {
                    font-weight: 600;
                    margin-bottom: 8px;
                    color: var(--ll-blue);
                    display: block;
                    font-size: 14px;
                }
                
                .guide-text {
                    font-size: 13px;
                    color: var(--ll-gray-600);
                    margin: 0;
                    line-height: 1.5;
                }
                
                @keyframes pulse {
                    0% { transform: scale(0.95); box-shadow: 0 0 0 0 rgba(18, 183, 106, 0.7); }
                    70% { transform: scale(1); box-shadow: 0 0 0 6px rgba(18, 183, 106, 0); }
                    100% { transform: scale(0.95); box-shadow: 0 0 0 0 rgba(18, 183, 106, 0); }
                }

                /* Responsive */
                @media (max-width: 768px) {
                    .ll-grid, .guide-grid {
                        grid-template-columns: 1fr;
                    }
                    .loadlink-hero {
                        flex-direction: column;
                        text-align: center;
                    }
                    .hero-logo-container {
                        margin-top: 24px;
                    }
                    .status-badge {
                        justify-content: flex-end;
                    }
                }
            </style>

            <div class="loadlink-hero">
                <div class="hero-content">
                    <h1>Loadlink Parcel Freight for WooCommerce</h1>
                    <p class="hero-subtitle">Seamlessly manage your freight orders and carrier settings.</p>
                </div>
                <div class="hero-logo-container">
                   <svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px"
                         viewBox="0 0 689 101.9" style="enable-background:new 0 0 689 101.9;" xml:space="preserve" width="200" height="30">
                    <style type="text/css">
                        .st0{fill:#1E3D5B;}
                        .st1{fill:#F37D48;}
                    </style>
                    <g>
                        <g>
                            <path class="st0" d="M193.6,77.5c-1.5-1-1.6-2.5-0.2-4.5l18.8-28.4c1.4-2,3.4-3.5,6.2-4.5c2.7-0.9,6.4-1.4,10.9-1.4l40,0.1
                                c4.6,0,7.6,0.5,9,1.4c1.4,1,1.4,2.5,0,4.5l-18.8,28.4c-1.4,2-3.4,3.5-6,4.5c-2.6,0.9-6.3,1.4-10.8,1.4l-40-0.1
                                C198.1,79,195.1,78.5,193.6,77.5z M223.1,69.2c0.6,0.4,1.8,0.6,3.7,0.6l3.5,0c2,0,3.5-0.2,4.6-0.6c1.1-0.4,2-1.2,2.8-2.4
                                l10.5-15.9c0.8-1.2,0.9-2,0.4-2.4c-0.5-0.4-1.8-0.6-3.7-0.6h-3.5c-1.9,0-3.4,0.2-4.6,0.6c-1.1,0.4-2.1,1.2-2.9,2.4l-10.5,15.9
                                C222.6,68,222.5,68.8,223.1,69.2z"/>
                            <polygon class="st1" points="458,69.9 478.6,38.8 452.7,38.8 426,79 470.1,79 481.1,69.9 		"/>
                            <path class="st0" d="M434.4,40.2c-1.4-1-4.3-1.5-9-1.5l-58.9,0l-11,9.2l11,0L345.9,79l52.9,0.1c4.5,0,8.1-0.5,10.8-1.4
                                c2.7-0.9,4.7-2.4,6.1-4.5l18.8-28.4C435.9,42.7,435.9,41.2,434.4,40.2z M404.5,51L394,66.9c-0.8,1.2-1.7,2-2.8,2.4
                                c-1.1,0.4-2.6,0.6-4.6,0.6l-8.6,0l14.6-22h8.6c2,0,3.2,0.3,3.8,0.7C405.4,49,405.3,49.8,404.5,51z"/>
                            <path class="st1" d="M563.6,79L552,64l-9.9,15l-25.9,0l26.6-40.2h18.7l11.6,15.1l10-15.1H609L582.4,79H563.6z"/>
                            <polygon class="st1" points="534.3,38.8 508.4,38.8 481.7,79 507.6,79 		"/>
                            <polygon class="st1" points="651.3,58.9 683.7,38.8 657.8,38.8 633.2,54.3 643.4,38.8 617.5,38.8 590.8,79 616.7,79 627,63.6 
                                631.1,79 656.9,79 		"/>
                            <polygon class="st0" points="343.8,38.8 307.8,38.8 261.6,79 300.4,79 311.4,69.9 297.3,69.9 317.9,51.8 314.6,79 339.9,79 		"/>
                            <polygon class="st0" points="165.7,69.9 186.2,38.8 160.3,38.8 133.7,79 177.8,79 188.8,69.9 		"/>
                        </g>
                        <g>
                            <g>
                                <path class="st1" d="M119.2,22.9c-6.6,1.8-13.5,4.1-20.7,6.9l6.9,1.4c5.5-2.6,11.1-5,16.7-7.2L119.2,22.9z"/>
                                <path class="st1" d="M45.4,58.2H62c6.5-5.2,13.5-10.1,20.9-14.7l-10.2-1.9C63.5,46.5,54.4,52,45.4,58.2z"/>
                                <path class="st1" d="M134.9,19.5c-3.4,0.5-6.9,1.2-10.6,2.1l2.1,0.8C129.3,21.3,132.1,20.4,134.9,19.5z"/>
                                <path class="st1" d="M77.8,39l9.7,1.7c4.4-2.6,8.9-5,13.5-7.3l-7.5-1.6C88.3,33.9,83.1,36.3,77.8,39z"/>
                            </g>
                            <g>
                                <path class="st1" d="M38.1,63.4c-0.5,0.4-1,0.7-1.5,1.1C25.3,73.2,14.3,82.9,4.4,93.7h24c4.9-7.3,17.3-20.9,27.3-30.3H38.1z"/>
                            </g>
                            <path class="st0" d="M64.2,93.7c14.6-24.7,35.9-47.8,69.9-67.8c-45.7,12.5-84,55.4-91.5,67.8H64.2z"/>
                            <g>
                                <path class="st0" d="M120.3,34.4c-3,2-6,4-8.8,6.1c1,3.2,1.5,6.5,1.5,10c0,18.9-15.3,34.1-34.1,34.1c-0.6,0-1.1,0-1.7,0
                                    c-2.4,3.1-4.6,6.2-6.7,9.5l-0.1,0.1c2.7,0.5,5.5,0.8,8.4,0.8c24.6,0,44.5-19.9,44.5-44.5C123.4,44.9,122.3,39.5,120.3,34.4z"/>
                            </g>
                            <g>
                                <g>
                                    <path class="st0" d="M78.9,6.1C54.3,6.1,34.4,26,34.4,50.6c0,0.7,0,1.3,0,2c3.5-2.2,7-4.2,10.6-6.2
                                        c2.1-16.9,16.5-29.9,33.9-29.9c4.1,0,8,0.7,11.6,2l15.3-3.3C98.4,9.5,89,6.1,78.9,6.1z"/>
                                </g>
                            </g>
                            <path class="st0" d="M118.3,18.9c-16.1,1.5-40.9,7.4-50,9.9c-8.9,2.5-16.2,5.1-20.9,6.8c-1.7,3.4-12.6,13-13,17
                                c9.6-5.7,25-13.2,35.8-17.8c9.2-4,27.1-10,30.3-10.9C103.8,22.9,118.3,18.9,118.3,18.9z"/>
                        </g>
                    </g>
                    </svg>
                </div>
            </div>

            <div class="ll-grid">
                <!-- App Status -->
                <div class="ll-card">
                    <div class="ll-card-header">
                        <h2 class="ll-card-title">App Status</h2>
                        <span class="ll-card-subtitle">System Health</span>
                    </div>
                    
                    <div class="stat-row">
                        <span class="stat-label">API Configuration</span>
                        <div class="status-badge <?php echo $is_configured ? 'status-active' : 'status-pending'; ?>">
                            <span class="status-dot <?php echo $is_configured ? 'dot-active' : 'dot-pending'; ?>"></span>
                            <?php echo $is_configured ? 'Active' : 'Pending'; ?>
                        </div>
                    </div>
                    
                    <div class="stat-row">
                        <span class="stat-label">Freight Orders</span>
                        <span class="stat-value"><?php echo number_format($freight_count); ?></span>
                    </div>
                </div>

                <!-- Quick Actions -->
                <div class="ll-card">
                    <div class="ll-card-header">
                        <h2 class="ll-card-title">Quick Actions</h2>
                    </div>
                    
                    <a href="<?php echo admin_url('admin.php?page=loadlink-settings'); ?>" class="ll-btn ll-btn-primary">Configure Settings</a>
                    <a href="<?php echo admin_url('admin.php?page=wc-settings&tab=shipping'); ?>" class="ll-btn ll-btn-secondary">Setup Carrier Service</a>
                    <a href="<?php echo admin_url('admin.php?page=loadlink-freight'); ?>" class="ll-btn ll-btn-secondary">View Orders</a>
                </div>
            </div>

            <div class="ll-card">
                <div class="ll-card-header">
                    <h2 class="ll-card-title">Getting Started Guide</h2>
                </div>
                
                <div class="guide-grid">
                    <div class="guide-item">
                        <span class="guide-title">1. Connect API</span>
                        <p class="guide-text">Go to Settings and enter your Loadlink credentials (API Key, Secret, User ID).</p>
                    </div>
                    <div class="guide-item">
                        <span class="guide-title">2. Register Carrier</span>
                        <p class="guide-text">Use the Setup page to register the app as a carrier service for checkout rates.</p>
                    </div>
                    <div class="guide-item">
                        <span class="guide-title">3. Test & Launch</span>
                        <p class="guide-text">Add products to cart and checkout to verify shipping rates appear correctly.</p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    public static function about_page() {
        ?>
        <div class="wrap">
            <h1 class="screen-reader-text">About Loadlink for WooCommerce</h1>
            <hr class="wp-header-end">
            
            <div class="loadlink-header">
                <div class="loadlink-logo">
                    <svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px"
                         viewBox="0 0 689 101.9" style="enable-background:new 0 0 689 101.9;" xml:space="preserve" width="200" height="30">
                    <style type="text/css">
                        .st0{fill:#1E3D5B;}
                        .st1{fill:#F37D48;}
                    </style>
                    <g>
                        <g>
                            <path class="st0" d="M193.6,77.5c-1.5-1-1.6-2.5-0.2-4.5l18.8-28.4c1.4-2,3.4-3.5,6.2-4.5c2.7-0.9,6.4-1.4,10.9-1.4l40,0.1
                                c4.6,0,7.6,0.5,9,1.4c1.4,1,1.4,2.5,0,4.5l-18.8,28.4c-1.4,2-3.4,3.5-6,4.5c-2.6,0.9-6.3,1.4-10.8,1.4l-40-0.1
                                C198.1,79,195.1,78.5,193.6,77.5z M223.1,69.2c0.6,0.4,1.8,0.6,3.7,0.6l3.5,0c2,0,3.5-0.2,4.6-0.6c1.1-0.4,2-1.2,2.8-2.4
                                l10.5-15.9c0.8-1.2,0.9-2,0.4-2.4c-0.5-0.4-1.8-0.6-3.7-0.6h-3.5c-1.9,0-3.4,0.2-4.6,0.6c-1.1,0.4-2.1,1.2-2.9,2.4l-10.5,15.9
                                C222.6,68,222.5,68.8,223.1,69.2z"/>
                            <polygon class="st1" points="458,69.9 478.6,38.8 452.7,38.8 426,79 470.1,79 481.1,69.9 		"/>
                            <path class="st0" d="M434.4,40.2c-1.4-1-4.3-1.5-9-1.5l-58.9,0l-11,9.2l11,0L345.9,79l52.9,0.1c4.5,0,8.1-0.5,10.8-1.4
                                c2.7-0.9,4.7-2.4,6.1-4.5l18.8-28.4C435.9,42.7,435.9,41.2,434.4,40.2z M404.5,51L394,66.9c-0.8,1.2-1.7,2-2.8,2.4
                                c-1.1,0.4-2.6,0.6-4.6,0.6l-8.6,0l14.6-22h8.6c2,0,3.2,0.3,3.8,0.7C405.4,49,405.3,49.8,404.5,51z"/>
                            <path class="st1" d="M563.6,79L552,64l-9.9,15l-25.9,0l26.6-40.2h18.7l11.6,15.1l10-15.1H609L582.4,79H563.6z"/>
                            <polygon class="st1" points="534.3,38.8 508.4,38.8 481.7,79 507.6,79 		"/>
                            <polygon class="st1" points="651.3,58.9 683.7,38.8 657.8,38.8 633.2,54.3 643.4,38.8 617.5,38.8 590.8,79 616.7,79 627,63.6 
                                631.1,79 656.9,79 		"/>
                            <polygon class="st0" points="343.8,38.8 307.8,38.8 261.6,79 300.4,79 311.4,69.9 297.3,69.9 317.9,51.8 314.6,79 339.9,79 		"/>
                            <polygon class="st0" points="165.7,69.9 186.2,38.8 160.3,38.8 133.7,79 177.8,79 188.8,69.9 		"/>
                        </g>
                        <g>
                            <g>
                                <path class="st1" d="M119.2,22.9c-6.6,1.8-13.5,4.1-20.7,6.9l6.9,1.4c5.5-2.6,11.1-5,16.7-7.2L119.2,22.9z"/>
                                <path class="st1" d="M45.4,58.2H62c6.5-5.2,13.5-10.1,20.9-14.7l-10.2-1.9C63.5,46.5,54.4,52,45.4,58.2z"/>
                                <path class="st1" d="M134.9,19.5c-3.4,0.5-6.9,1.2-10.6,2.1l2.1,0.8C129.3,21.3,132.1,20.4,134.9,19.5z"/>
                                <path class="st1" d="M77.8,39l9.7,1.7c4.4-2.6,8.9-5,13.5-7.3l-7.5-1.6C88.3,33.9,83.1,36.3,77.8,39z"/>
                            </g>
                            <g>
                                <path class="st1" d="M38.1,63.4c-0.5,0.4-1,0.7-1.5,1.1C25.3,73.2,14.3,82.9,4.4,93.7h24c4.9-7.3,17.3-20.9,27.3-30.3H38.1z"/>
                            </g>
                            <path class="st0" d="M64.2,93.7c14.6-24.7,35.9-47.8,69.9-67.8c-45.7,12.5-84,55.4-91.5,67.8H64.2z"/>
                            <g>
                                <path class="st0" d="M120.3,34.4c-3,2-6,4-8.8,6.1c1,3.2,1.5,6.5,1.5,10c0,18.9-15.3,34.1-34.1,34.1c-0.6,0-1.1,0-1.7,0
                                    c-2.4,3.1-4.6,6.2-6.7,9.5l-0.1,0.1c2.7,0.5,5.5,0.8,8.4,0.8c24.6,0,44.5-19.9,44.5-44.5C123.4,44.9,122.3,39.5,120.3,34.4z"/>
                            </g>
                            <g>
                                <g>
                                    <path class="st0" d="M78.9,6.1C54.3,6.1,34.4,26,34.4,50.6c0,0.7,0,1.3,0,2c3.5-2.2,7-4.2,10.6-6.2
                                        c2.1-16.9,16.5-29.9,33.9-29.9c4.1,0,8,0.7,11.6,2l15.3-3.3C98.4,9.5,89,6.1,78.9,6.1z"/>
                                </g>
                            </g>
                            <path class="st0" d="M118.3,18.9c-16.1,1.5-40.9,7.4-50,9.9c-8.9,2.5-16.2,5.1-20.9,6.8c-1.7,3.4-12.6,13-13,17
                                c9.6-5.7,25-13.2,35.8-17.8c9.2-4,27.1-10,30.3-10.9C103.8,22.9,118.3,18.9,118.3,18.9z"/>
                        </g>
                    </g>
                    </svg>
                </div>
                <h1>About</h1>
            </div>
            
            <div class="loadlink-about">
                <div class="loadlink-logo-section">
                    <h2>Loadlink</h2>
                    <p><strong>Version:</strong> <?php echo LOADLINK_VERSION; ?></p>
                    <p><strong>Author:</strong> Loadlink</p>
                    <p><strong>Description:</strong> Provides live shipping rates from the Loadlink platform.</p>
                </div>
                
                <div class="loadlink-features">
                    <h2>Features</h2>
                    <ul>
                        <li>Real-time shipping rate calculation</li>
                        <li>Multiple shipping service options</li>
                        <li>Product-specific packaging types</li>
                        <li>Building type selection (Residential/Commercial)</li>
                        <li>Intelligent caching to prevent duplicate API calls</li>
                        <li>WooCommerce Blocks compatibility</li>
                        <li>Comprehensive logging and debugging</li>
                    </ul>
                </div>
                
                <div class="loadlink-requirements">
                    <h2>Requirements</h2>
                    <ul>
                        <li>WordPress 5.0 or higher</li>
                        <li>WooCommerce 3.0 or higher</li>
                        <li>PHP 7.4 or higher</li>
                        <li>Loadlink API credentials</li>
                    </ul>
                </div>
                
                <div class="loadlink-support">
                    <h2>Support</h2>
                    <p>For technical support and documentation, please visit the <a href="https://www.loadlink.com.au/help/" target="_blank">Loadlink website</a>.</p>
                </div>
            </div>
            
            <style>
            .loadlink-about {
                max-width: 800px;
            }
            .loadlink-logo-section,
            .loadlink-features,
            .loadlink-requirements,
            .loadlink-support {
                background: #fff;
                border: 1px solid #ccd0d4;
                border-radius: 4px;
                padding: 20px;
                margin: 20px 0;
            }
            .loadlink-about h2 {
                margin-top: 0;
                color: #23282d;
            }
            .loadlink-about ul {
                margin-left: 20px;
            }
            </style>
        </div>
        <?php
    }
    
    public static function instructions_page() {
        ?>
        <div class="wrap">
            <h1 class="screen-reader-text">Loadlink Setup Instructions</h1>
            <hr class="wp-header-end">

            <div class="loadlink-header">
                <div class="loadlink-logo">
                    <svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px"
                         viewBox="0 0 689 101.9" style="enable-background:new 0 0 689 101.9;" xml:space="preserve" width="200" height="30">
                    <style type="text/css">
                        .st0{fill:#1E3D5B;}
                        .st1{fill:#F37D48;}
                    </style>
                    <g>
                        <g>
                            <path class="st0" d="M193.6,77.5c-1.5-1-1.6-2.5-0.2-4.5l18.8-28.4c1.4-2,3.4-3.5,6.2-4.5c2.7-0.9,6.4-1.4,10.9-1.4l40,0.1
                                c4.6,0,7.6,0.5,9,1.4c1.4,1,1.4,2.5,0,4.5l-18.8,28.4c-1.4,2-3.4,3.5-6,4.5c-2.6,0.9-6.3,1.4-10.8,1.4l-40-0.1
                                C198.1,79,195.1,78.5,193.6,77.5z M223.1,69.2c0.6,0.4,1.8,0.6,3.7,0.6l3.5,0c2,0,3.5-0.2,4.6-0.6c1.1-0.4,2-1.2,2.8-2.4
                                l10.5-15.9c0.8-1.2,0.9-2,0.4-2.4c-0.5-0.4-1.8-0.6-3.7-0.6h-3.5c-1.9,0-3.4,0.2-4.6,0.6c-1.1,0.4-2.1,1.2-2.9,2.4l-10.5,15.9
                                C222.6,68,222.5,68.8,223.1,69.2z"/>
                            <polygon class="st1" points="458,69.9 478.6,38.8 452.7,38.8 426,79 470.1,79 481.1,69.9 		"/>
                            <path class="st0" d="M434.4,40.2c-1.4-1-4.3-1.5-9-1.5l-58.9,0l-11,9.2l11,0L345.9,79l52.9,0.1c4.5,0,8.1-0.5,10.8-1.4
                                c2.7-0.9,4.7-2.4,6.1-4.5l18.8-28.4C435.9,42.7,435.9,41.2,434.4,40.2z M404.5,51L394,66.9c-0.8,1.2-1.7,2-2.8,2.4
                                c-1.1,0.4-2.6,0.6-4.6,0.6l-8.6,0l14.6-22h8.6c2,0,3.2,0.3,3.8,0.7C405.4,49,405.3,49.8,404.5,51z"/>
                            <path class="st1" d="M563.6,79L552,64l-9.9,15l-25.9,0l26.6-40.2h18.7l11.6,15.1l10-15.1H609L582.4,79H563.6z"/>
                            <polygon class="st1" points="534.3,38.8 508.4,38.8 481.7,79 507.6,79 		"/>
                            <polygon class="st1" points="651.3,58.9 683.7,38.8 657.8,38.8 633.2,54.3 643.4,38.8 617.5,38.8 590.8,79 616.7,79 627,63.6 
                                631.1,79 656.9,79 		"/>
                            <polygon class="st0" points="343.8,38.8 307.8,38.8 261.6,79 300.4,79 311.4,69.9 297.3,69.9 317.9,51.8 314.6,79 339.9,79 		"/>
                            <polygon class="st0" points="165.7,69.9 186.2,38.8 160.3,38.8 133.7,79 177.8,79 188.8,69.9 		"/>
                        </g>
                        <g>
                            <g>
                                <path class="st1" d="M119.2,22.9c-6.6,1.8-13.5,4.1-20.7,6.9l6.9,1.4c5.5-2.6,11.1-5,16.7-7.2L119.2,22.9z"/>
                                <path class="st1" d="M45.4,58.2H62c6.5-5.2,13.5-10.1,20.9-14.7l-10.2-1.9C63.5,46.5,54.4,52,45.4,58.2z"/>
                                <path class="st1" d="M134.9,19.5c-3.4,0.5-6.9,1.2-10.6,2.1l2.1,0.8C129.3,21.3,132.1,20.4,134.9,19.5z"/>
                                <path class="st1" d="M77.8,39l9.7,1.7c4.4-2.6,8.9-5,13.5-7.3l-7.5-1.6C88.3,33.9,83.1,36.3,77.8,39z"/>
                            </g>
                            <g>
                                <path class="st1" d="M38.1,63.4c-0.5,0.4-1,0.7-1.5,1.1C25.3,73.2,14.3,82.9,4.4,93.7h24c4.9-7.3,17.3-20.9,27.3-30.3H38.1z"/>
                            </g>
                            <path class="st0" d="M64.2,93.7c14.6-24.7,35.9-47.8,69.9-67.8c-45.7,12.5-84,55.4-91.5,67.8H64.2z"/>
                            <g>
                                <path class="st0" d="M120.3,34.4c-3,2-6,4-8.8,6.1c1,3.2,1.5,6.5,1.5,10c0,18.9-15.3,34.1-34.1,34.1c-0.6,0-1.1,0-1.7,0
                                    c-2.4,3.1-4.6,6.2-6.7,9.5l-0.1,0.1c2.7,0.5,5.5,0.8,8.4,0.8c24.6,0,44.5-19.9,44.5-44.5C123.4,44.9,122.3,39.5,120.3,34.4z"/>
                            </g>
                            <g>
                                <g>
                                    <path class="st0" d="M78.9,6.1C54.3,6.1,34.4,26,34.4,50.6c0,0.7,0,1.3,0,2c3.5-2.2,7-4.2,10.6-6.2
                                        c2.1-16.9,16.5-29.9,33.9-29.9c4.1,0,8,0.7,11.6,2l15.3-3.3C98.4,9.5,89,6.1,78.9,6.1z"/>
                                </g>
                            </g>
                            <path class="st0" d="M118.3,18.9c-16.1,1.5-40.9,7.4-50,9.9c-8.9,2.5-16.2,5.1-20.9,6.8c-1.7,3.4-12.6,13-13,17
                                c9.6-5.7,25-13.2,35.8-17.8c9.2-4,27.1-10,30.3-10.9C103.8,22.9,118.3,18.9,118.3,18.9z"/>
                        </g>
                    </g>
                    </svg>
                </div>
                <h1>Instructions</h1>
            </div>
            
            <div class="loadlink-instructions">
                <div class="instruction-section">
                    <h2>1. API Configuration</h2>
                    <ol>
                        <li>Go to <strong>Loadlink → Settings</strong></li>
                        <li>Enter your Loadlink API credentials:
                            <ul>
                                <li><strong>API Key:</strong> Your Loadlink API public key</li>
                                <li><strong>API Secret:</strong> Your Loadlink API secret key</li>
                                <li><strong>User ID:</strong> Your Loadlink account user ID</li>
                            </ul>
                        </li>
                        <li>Configure building types for pickup and delivery</li>
                        <li>Set default packaging code (e.g., BX for Box)</li>
                        <li>Click <strong>Save Changes</strong></li>
                    </ol>
                </div>
                
                <div class="instruction-section">
                    <h2>2. Product Setup</h2>
                    <ol>
                        <li>Edit your products in <strong>Products → All Products</strong></li>
                        <li>For each product, go to the <strong>Shipping</strong> tab</li>
                        <li>Ensure all products have:
                            <ul>
                                <li><strong>Weight:</strong> Set in kg</li>
                                <li><strong>Dimensions:</strong> Length, width, height in cm</li>
                                <li><strong>Packaging Type:</strong> Select from dropdown (optional)</li>
                            </ul>
                        </li>
                        <li>Save the product</li>
                    </ol>
                </div>
                
                <div class="instruction-section">
                    <h2>3. Shipping Zone Configuration</h2>
                    <ol>
                        <li>Go to <strong>WooCommerce → Settings → Shipping</strong></li>
                        <li>Create or edit a shipping zone</li>
                        <li>Add <strong>Loadlink Live Rates</strong> as a shipping method</li>
                        <li>Configure the method settings if needed</li>
                        <li>Save changes</li>
                    </ol>
                </div>
                
                <div class="instruction-section">
                    <h2>4. Testing</h2>
                    <ol>
                        <li>Go to <strong>Loadlink → Settings</strong></li>
                        <li>Use the <strong>Test API with Real Data</strong> section</li>
                        <li>Enter a test destination address</li>
                        <li>Click <strong>Test API with Real Data</strong></li>
                        <li>Verify that rates are returned successfully</li>
                    </ol>
                </div>
                
                <div class="instruction-section">
                    <h2>5. Troubleshooting</h2>
                    <ul>
                        <li>Check <strong>WooCommerce → Status → Logs</strong> for Loadlink entries</li>
                        <li>Ensure all products have dimensions and weight</li>
                        <li>Verify API credentials are correct</li>
                        <li>Check that your store address is configured in WooCommerce settings</li>
                        <li>Use the <strong>Notices</strong> page to identify products with missing data</li>
                    </ul>
                </div>
            </div>
            
            <style>
            .loadlink-instructions {
                max-width: 900px;
            }
            .instruction-section {
                background: #fff;
                border: 1px solid #ccd0d4;
                border-radius: 4px;
                padding: 20px;
                margin: 20px 0;
            }
            .instruction-section h2 {
                margin-top: 0;
                color: #23282d;
            }
            .instruction-section ol,
            .instruction-section ul {
                margin-left: 20px;
            }
            .instruction-section li {
                margin-bottom: 8px;
            }
            </style>
        </div>
        <?php
    }
    
    public static function notices_page() {
        // Get products with missing dimensions or weight
        $products_with_issues = [];
        $products = get_posts([
            'post_type' => 'product',
            'posts_per_page' => -1,
            'post_status' => 'publish'
        ]);
        
        foreach ($products as $product_post) {
            $product = wc_get_product($product_post->ID);
            if (!$product) continue;
            
            $issues = [];
            $current_data = [
                'weight' => $product->get_weight(),
                'length' => $product->get_length(),
                'width' => $product->get_width(),
                'height' => $product->get_height(),
                'packaging_code' => $product->get_meta('_loadlink_packaging_code')
            ];
            
            if (empty($current_data['weight']) || $current_data['weight'] <= 0) {
                $issues[] = 'Missing or invalid weight';
            }
            if (empty($current_data['length']) || $current_data['length'] <= 0) {
                $issues[] = 'Missing or invalid length';
            }
            if (empty($current_data['width']) || $current_data['width'] <= 0) {
                $issues[] = 'Missing or invalid width';
            }
            if (empty($current_data['height']) || $current_data['height'] <= 0) {
                $issues[] = 'Missing or invalid height';
            }
            
            if (!empty($issues)) {
                $products_with_issues[] = [
                    'id' => $product->get_id(),
                    'name' => $product->get_name(),
                    'issues' => $issues,
                    'current_data' => $current_data
                ];
            }
        }
        
        ?>
        <div class="wrap">
            <h1 class="screen-reader-text">Loadlink Notices</h1>
            <hr class="wp-header-end">

            <div class="loadlink-header">
                <div class="loadlink-logo">
                    <svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px"
                         viewBox="0 0 689 101.9" style="enable-background:new 0 0 689 101.9;" xml:space="preserve" width="200" height="30">
                    <style type="text/css">
                        .st0{fill:#1E3D5B;}
                        .st1{fill:#F37D48;}
                    </style>
                    <g>
                        <g>
                            <path class="st0" d="M193.6,77.5c-1.5-1-1.6-2.5-0.2-4.5l18.8-28.4c1.4-2,3.4-3.5,6.2-4.5c2.7-0.9,6.4-1.4,10.9-1.4l40,0.1
                                c4.6,0,7.6,0.5,9,1.4c1.4,1,1.4,2.5,0,4.5l-18.8,28.4c-1.4,2-3.4,3.5-6,4.5c-2.6,0.9-6.3,1.4-10.8,1.4l-40-0.1
                                C198.1,79,195.1,78.5,193.6,77.5z M223.1,69.2c0.6,0.4,1.8,0.6,3.7,0.6l3.5,0c2,0,3.5-0.2,4.6-0.6c1.1-0.4,2-1.2,2.8-2.4
                                l10.5-15.9c0.8-1.2,0.9-2,0.4-2.4c-0.5-0.4-1.8-0.6-3.7-0.6h-3.5c-1.9,0-3.4,0.2-4.6,0.6c-1.1,0.4-2.1,1.2-2.9,2.4l-10.5,15.9
                                C222.6,68,222.5,68.8,223.1,69.2z"/>
                            <polygon class="st1" points="458,69.9 478.6,38.8 452.7,38.8 426,79 470.1,79 481.1,69.9 		"/>
                            <path class="st0" d="M434.4,40.2c-1.4-1-4.3-1.5-9-1.5l-58.9,0l-11,9.2l11,0L345.9,79l52.9,0.1c4.5,0,8.1-0.5,10.8-1.4
                                c2.7-0.9,4.7-2.4,6.1-4.5l18.8-28.4C435.9,42.7,435.9,41.2,434.4,40.2z M404.5,51L394,66.9c-0.8,1.2-1.7,2-2.8,2.4
                                c-1.1,0.4-2.6,0.6-4.6,0.6l-8.6,0l14.6-22h8.6c2,0,3.2,0.3,3.8,0.7C405.4,49,405.3,49.8,404.5,51z"/>
                            <path class="st1" d="M563.6,79L552,64l-9.9,15l-25.9,0l26.6-40.2h18.7l11.6,15.1l10-15.1H609L582.4,79H563.6z"/>
                            <polygon class="st1" points="534.3,38.8 508.4,38.8 481.7,79 507.6,79 		"/>
                            <polygon class="st1" points="651.3,58.9 683.7,38.8 657.8,38.8 633.2,54.3 643.4,38.8 617.5,38.8 590.8,79 616.7,79 627,63.6 
                                631.1,79 656.9,79 		"/>
                            <polygon class="st0" points="343.8,38.8 307.8,38.8 261.6,79 300.4,79 311.4,69.9 297.3,69.9 317.9,51.8 314.6,79 339.9,79 		"/>
                            <polygon class="st0" points="165.7,69.9 186.2,38.8 160.3,38.8 133.7,79 177.8,79 188.8,69.9 		"/>
                        </g>
                        <g>
                            <g>
                                <path class="st1" d="M119.2,22.9c-6.6,1.8-13.5,4.1-20.7,6.9l6.9,1.4c5.5-2.6,11.1-5,16.7-7.2L119.2,22.9z"/>
                                <path class="st1" d="M45.4,58.2H62c6.5-5.2,13.5-10.1,20.9-14.7l-10.2-1.9C63.5,46.5,54.4,52,45.4,58.2z"/>
                                <path class="st1" d="M134.9,19.5c-3.4,0.5-6.9,1.2-10.6,2.1l2.1,0.8C129.3,21.3,132.1,20.4,134.9,19.5z"/>
                                <path class="st1" d="M77.8,39l9.7,1.7c4.4-2.6,8.9-5,13.5-7.3l-7.5-1.6C88.3,33.9,83.1,36.3,77.8,39z"/>
                            </g>
                            <g>
                                <path class="st1" d="M38.1,63.4c-0.5,0.4-1,0.7-1.5,1.1C25.3,73.2,14.3,82.9,4.4,93.7h24c4.9-7.3,17.3-20.9,27.3-30.3H38.1z"/>
                            </g>
                            <path class="st0" d="M64.2,93.7c14.6-24.7,35.9-47.8,69.9-67.8c-45.7,12.5-84,55.4-91.5,67.8H64.2z"/>
                            <g>
                                <path class="st0" d="M120.3,34.4c-3,2-6,4-8.8,6.1c1,3.2,1.5,6.5,1.5,10c0,18.9-15.3,34.1-34.1,34.1c-0.6,0-1.1,0-1.7,0
                                    c-2.4,3.1-4.6,6.2-6.7,9.5l-0.1,0.1c2.7,0.5,5.5,0.8,8.4,0.8c24.6,0,44.5-19.9,44.5-44.5C123.4,44.9,122.3,39.5,120.3,34.4z"/>
                            </g>
                            <g>
                                <g>
                                    <path class="st0" d="M78.9,6.1C54.3,6.1,34.4,26,34.4,50.6c0,0.7,0,1.3,0,2c3.5-2.2,7-4.2,10.6-6.2
                                        c2.1-16.9,16.5-29.9,33.9-29.9c4.1,0,8,0.7,11.6,2l15.3-3.3C98.4,9.5,89,6.1,78.9,6.1z"/>
                                </g>
                            </g>
                            <path class="st0" d="M118.3,18.9c-16.1,1.5-40.9,7.4-50,9.9c-8.9,2.5-16.2,5.1-20.9,6.8c-1.7,3.4-12.6,13-13,17
                                c9.6-5.7,25-13.2,35.8-17.8c9.2-4,27.1-10,30.3-10.9C103.8,22.9,118.3,18.9,118.3,18.9z"/>
                        </g>
                    </g>
                    </svg>
                </div>
                <h1>Notices</h1>
            </div>
            
            <div class="loadlink-notices">
                <?php if (empty($products_with_issues)): ?>
                    <div class="notice notice-success">
                        <p><strong>✅ All products have proper dimensions and weight configured.</strong></p>
                    </div>
                <?php else: ?>
                    <div class="notice notice-warning">
                        <p><strong>⚠️ The following products have missing or invalid shipping data:</strong></p>
                    </div>
                    
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th>Product</th>
                                <th>Issues</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($products_with_issues as $product): ?>
                                <tr>
                                    <td>
                                        <strong><?php echo esc_html($product['name']); ?></strong>
                                        <br>
                                        <small>ID: <?php echo $product['id']; ?></small>
                                    </td>
                                    <td>
                                        <ul style="margin: 0; padding-left: 20px;">
                                            <?php foreach ($product['issues'] as $issue): ?>
                                                <li><?php echo esc_html($issue); ?></li>
                                            <?php endforeach; ?>
                                        </ul>
                                    </td>
                                    <td>
                                        <a href="<?php echo admin_url('post.php?post=' . $product['id'] . '&action=edit'); ?>" class="button button-small">Edit Product</a>
                                        <button class="button button-small loadlink-quick-edit" data-product-id="<?php echo $product['id']; ?>" data-product-name="<?php echo esc_attr($product['name']); ?>" data-current-data="<?php echo esc_attr(json_encode($product['current_data'])); ?>">Quick Edit</button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
                
                <div class="loadlink-notice-info">
                    <h2>Why This Matters</h2>
                    <p>Loadlink requires accurate product dimensions and weight to calculate shipping rates. Products with missing or invalid data will:</p>
                    <ul>
                        <li>Prevent shipping rate calculation</li>
                        <li>Show error messages to customers</li>
                        <li>Reduce conversion rates</li>
                    </ul>
                </div>
            </div>
            
            <style>
            .loadlink-notices {
                max-width: 1000px;
            }
            .loadlink-notice-info {
                background: #fff;
                border: 1px solid #ccd0d4;
                border-radius: 4px;
                padding: 20px;
                margin: 20px 0;
            }
            .loadlink-notice-info h2 {
                margin-top: 0;
                color: #23282d;
            }
            </style>
            
            <!-- Quick Edit Modal -->
            <div id="loadlink-quick-edit-modal" style="display: none;">
                <div class="loadlink-modal-content">
                    <div class="loadlink-modal-header">
                        <h2>Quick Edit Product</h2>
                        <span class="loadlink-modal-close">&times;</span>
                    </div>
                    <div class="loadlink-modal-body">
                        <form id="loadlink-quick-edit-form">
                            <input type="hidden" id="quick-edit-product-id" name="product_id">
                            <table class="form-table">
                                <tr>
                                    <th scope="row">Weight (kg)</th>
                                    <td><input type="number" id="quick-edit-weight" name="weight" step="0.1" min="0" class="regular-text"></td>
                                </tr>
                                <tr>
                                    <th scope="row">Length (cm)</th>
                                    <td><input type="number" id="quick-edit-length" name="length" step="0.1" min="0" class="regular-text"></td>
                                </tr>
                                <tr>
                                    <th scope="row">Width (cm)</th>
                                    <td><input type="number" id="quick-edit-width" name="width" step="0.1" min="0" class="regular-text"></td>
                                </tr>
                                <tr>
                                    <th scope="row">Height (cm)</th>
                                    <td><input type="number" id="quick-edit-height" name="height" step="0.1" min="0" class="regular-text"></td>
                                </tr>
                                <tr>
                                    <th scope="row">Packaging Type</th>
                                    <td>
                                        <select id="quick-edit-packaging" name="packaging_code" class="regular-text">
                                            <option value="">Select Packaging Type</option>
                                            <option value="CR">Crate</option>
                                            <option value="DR">Drum</option>
                                            <option value="RO">Roll</option>
                                            <option value="BX">Box</option>
                                            <option value="EN">Envelope</option>
                                            <option value="SA">Satchel</option>
                                            <option value="PA">Pallet</option>
                                            <option value="BG">Bag</option>
                                            <option value="CT">Carton</option>
                                            <option value="PC">Piece</option>
                                        </select>
                                    </td>
                                </tr>
                            </table>
                        </form>
                    </div>
                    <div class="loadlink-modal-footer">
                        <button type="button" class="button" id="loadlink-quick-edit-cancel">Cancel</button>
                        <button type="button" class="button button-primary" id="loadlink-quick-edit-save">Save Changes</button>
                    </div>
                </div>
            </div>
            
            <script>
            jQuery(document).ready(function($) {
                // Open modal
                $('.loadlink-quick-edit').on('click', function() {
                    var productId = $(this).data('product-id');
                    var productName = $(this).data('product-name');
                    var currentData = $(this).data('current-data');
                    
                    $('#quick-edit-product-id').val(productId);
                    $('#quick-edit-weight').val(currentData.weight || '');
                    $('#quick-edit-length').val(currentData.length || '');
                    $('#quick-edit-width').val(currentData.width || '');
                    $('#quick-edit-height').val(currentData.height || '');
                    $('#quick-edit-packaging').val(currentData.packaging_code || '');
                    
                    $('.loadlink-modal-header h2').text('Quick Edit: ' + productName);
                    $('#loadlink-quick-edit-modal').show();
                });
                
                // Close modal
                $('.loadlink-modal-close, #loadlink-quick-edit-cancel').on('click', function() {
                    $('#loadlink-quick-edit-modal').hide();
                });
                
                // Save changes
                $('#loadlink-quick-edit-save').on('click', function() {
                    var formData = {
                        action: 'loadlink_quick_edit_product',
                        nonce: '<?php echo wp_create_nonce('loadlink_quick_edit'); ?>',
                        product_id: $('#quick-edit-product-id').val(),
                        weight: $('#quick-edit-weight').val(),
                        length: $('#quick-edit-length').val(),
                        width: $('#quick-edit-width').val(),
                        height: $('#quick-edit-height').val(),
                        packaging_code: $('#quick-edit-packaging').val()
                    };
                    
                    $.post(ajaxurl, formData, function(response) {
                        if (response.success) {
                            alert('Product updated successfully!');
                            location.reload();
                        } else {
                            alert('Error: ' + response.data.message);
                        }
                    });
                });
            });
            </script>
            
            <style>
            #loadlink-quick-edit-modal {
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0,0,0,0.5);
                z-index: 9999;
            }
            .loadlink-modal-content {
                position: absolute;
                top: 50%;
                left: 50%;
                transform: translate(-50%, -50%);
                background: #fff;
                border-radius: 4px;
                box-shadow: 0 4px 6px rgba(0,0,0,0.1);
                max-width: 500px;
                width: 90%;
                max-height: 80vh;
                overflow-y: auto;
            }
            .loadlink-modal-header {
                padding: 20px;
                border-bottom: 1px solid #ddd;
                display: flex;
                justify-content: space-between;
                align-items: center;
            }
            .loadlink-modal-header h2 {
                margin: 0;
            }
            .loadlink-modal-close {
                font-size: 24px;
                cursor: pointer;
                color: #999;
            }
            .loadlink-modal-body {
                padding: 20px;
            }
            .loadlink-modal-footer {
                padding: 20px;
                border-top: 1px solid #ddd;
                text-align: right;
            }
            .loadlink-modal-footer .button {
                margin-left: 10px;
            }
            </style>
        </div>
        <?php
    }
    
    public static function quick_edit_product_ajax() {
        try {
            check_ajax_referer('loadlink_quick_edit', 'nonce');
            
            if (!current_user_can('manage_options')) {
                wp_send_json_error(['message' => 'Unauthorized']);
                return;
            }
            
            $product_id = intval($_POST['product_id']);
            $weight = sanitize_text_field($_POST['weight']);
            $length = sanitize_text_field($_POST['length']);
            $width = sanitize_text_field($_POST['width']);
            $height = sanitize_text_field($_POST['height']);
            $packaging_code = sanitize_text_field($_POST['packaging_code']);
            
            if (!$product_id) {
                wp_send_json_error(['message' => 'Invalid product ID']);
                return;
            }
            
            $product = wc_get_product($product_id);
            if (!$product) {
                wp_send_json_error(['message' => 'Product not found']);
                return;
            }
            
            // Update product dimensions and weight
            $product->set_weight($weight);
            $product->set_length($length);
            $product->set_width($width);
            $product->set_height($height);
            $product->save();
            
            // Update packaging code meta
            if ($packaging_code) {
                $product->update_meta_data('_loadlink_packaging_code', $packaging_code);
                $product->save();
            } else {
                $product->delete_meta_data('_loadlink_packaging_code');
                $product->save();
            }
            
            wp_send_json_success(['message' => 'Product updated successfully']);
            
        } catch (Exception $e) {
            wp_send_json_error(['message' => 'Error updating product: ' . $e->getMessage()]);
        }
    }
    
    public static function admin_notices() {
        // Only show on admin dashboard and if user can manage options
        if (!current_user_can('manage_options') || !is_admin()) {
            return;
        }
        
        // Check if we're on the dashboard or Loadlink pages
        $screen = get_current_screen();
        if (!$screen || (!in_array($screen->id, ['dashboard', 'loadlink-dashboard', 'loadlink-notices']) && strpos($screen->id, 'loadlink') === false)) {
            return;
        }
        
        // Get products with missing data
        $products_with_issues = self::get_products_with_missing_data();
        
        if (!empty($products_with_issues)) {
            $count = count($products_with_issues);
            $notice_class = 'notice notice-warning is-dismissible';
            $message = sprintf(
                '<strong>Loadlink:</strong> %d product(s) have missing shipping data. <a href="%s">Fix now</a>',
                $count,
                admin_url('admin.php?page=loadlink-notices')
            );
            
            echo '<div class="' . esc_attr($notice_class) . '"><p>' . $message . '</p></div>';
        }
    }
    
    private static function get_products_with_missing_data() {
        $products_with_issues = [];
        $products = get_posts([
            'post_type' => 'product',
            'posts_per_page' => -1,
            'post_status' => 'publish'
        ]);
        
        foreach ($products as $product_post) {
            $product = wc_get_product($product_post->ID);
            if (!$product) continue;
            
            $has_issues = false;
            if (empty($product->get_weight()) || $product->get_weight() <= 0) {
                $has_issues = true;
            }
            if (empty($product->get_length()) || $product->get_length() <= 0) {
                $has_issues = true;
            }
            if (empty($product->get_width()) || $product->get_width() <= 0) {
                $has_issues = true;
            }
            if (empty($product->get_height()) || $product->get_height() <= 0) {
                $has_issues = true;
            }
            
            if ($has_issues) {
                $products_with_issues[] = $product->get_id();
            }
        }
        
        return $products_with_issues;
    }
}
