<?php
class Loadlink_Rates_Controller {
    private static $last_calculation_time = 0;
    private static $last_package_hash = '';

    public static function get_rates($package) {
        // Guard: avoid API calls until destination looks complete
        $dest = isset($package['destination']) ? $package['destination'] : [];
        $has_postcode = !empty($dest['postcode']);
        $has_state = !empty($dest['state']);
        $has_country = !empty($dest['country']);
        $has_city = !empty($dest['city']);

        if (!$has_postcode || !$has_state || !$has_country || !$has_city) {
            Loadlink_Logger::log('Incomplete address - missing required fields', 'debug');
            return [];
        }

        // Duplicate-call prevention
        $package_hash = md5(serialize($package));
        $now = time();
        $throttle_window = 5; // seconds

        if (self::$last_package_hash === $package_hash &&
            ($now - self::$last_calculation_time) < $throttle_window) {
            Loadlink_Logger::log('Duplicate rate calculation prevented - same package within throttle window', 'debug');
            return [];
        }

        // Record this calculation
        self::$last_calculation_time = $now;
        self::$last_package_hash     = $package_hash;

        // Build final payload
        $payload = self::build_payload($package);

        // Call API client
        $client = new Loadlink_Client();
        return $client->fetch_rates($payload);
    }

    /**
     * Prepare payload for API request
     * Enforce minimum total order weight of 1 kg
     */
    private static function build_payload($package) {

        $total_weight = 0;

        if (!empty($package['contents'])) {
            foreach ($package['contents'] as $item) {
                if (isset($item['data']) && is_object($item['data'])) {
                    $weight = (float) $item['data']->get_weight();
                    $qty    = isset($item['quantity']) ? (int) $item['quantity'] : 1;
                    $total_weight += ($weight * $qty);
                }
            }
        }

        // Convert grams → kilograms if needed
        $weight_unit = get_option('woocommerce_weight_unit', 'kg');
        if ($weight_unit === 'g') {
            $total_weight = $total_weight / 1000;
        }

        // Enforce minimum 1 kg
        if ($total_weight < 1) {
            Loadlink_Logger::log('Adjusted total weight to minimum 1 kg (original: ' . $total_weight . ')', 'debug');
            $total_weight = 1;
        }

        // Store the adjusted total in payload
        $package['loadlink_total_weight'] = $total_weight;

        return $package;
    }
}
