<?php
class Loadlink_Plugin {
    public function init() {
        // Only initialize if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', [$this, 'woocommerce_missing_notice']);
            return;
        }
        
        // Load core classes first
        require_once LOADLINK_PLUGIN_PATH . 'includes/class-loadlink-logger.php';
        require_once LOADLINK_PLUGIN_PATH . 'includes/class-loadlink-client.php';
        require_once LOADLINK_PLUGIN_PATH . 'includes/class-loadlink-settings.php';
        require_once LOADLINK_PLUGIN_PATH . 'includes/class-loadlink-order-confirmation.php';
        require_once LOADLINK_PLUGIN_PATH . 'includes/class-loadlink-admin-freight.php';
        require_once LOADLINK_PLUGIN_PATH . 'includes/class-loadlink-admin-dashboard.php';
        require_once LOADLINK_PLUGIN_PATH . 'includes/class-loadlink-order-actions.php';
        require_once LOADLINK_PLUGIN_PATH . 'includes/class-loadlink-priority.php';
        
        add_action('woocommerce_shipping_init', [$this, 'load_shipping']);
        add_filter('woocommerce_shipping_methods', [$this, 'register_shipping']);
        Loadlink_Settings::init();
        Loadlink_Order_Confirmation::init();
        
        // Initialize admin management
        new Loadlink_Admin_Dashboard();
        new Loadlink_Admin_Freight();
        new Loadlink_Order_Actions();

		// Checkout field: Dropoff Building Type
		add_filter('woocommerce_checkout_fields', [$this, 'add_dropoff_building_type_field']);
		add_action('woocommerce_checkout_update_order_review', [$this, 'capture_dropoff_building_type_session']);
		add_action('woocommerce_checkout_update_order_meta', [$this, 'save_dropoff_building_type_order_meta']);
		
		// Checkout field: Dropoff Phone (required)
		add_filter('woocommerce_checkout_fields', [$this, 'add_dropoff_phone_field']);
		add_action('woocommerce_checkout_process', [$this, 'validate_dropoff_phone_field']);
		add_action('woocommerce_checkout_update_order_meta', [$this, 'save_dropoff_phone_order_meta']);
		
		add_action('woocommerce_checkout_posted_data', [$this, 'sync_billing_to_shipping'], 5);


		// Make native WooCommerce shipping phone required and full-width (classic checkout)
		add_filter('woocommerce_checkout_fields', [$this, 'require_shipping_phone_field'], 20);
		add_action('woocommerce_checkout_process', [$this, 'validate_native_shipping_phone']);
		
		// Capture selected Loadlink shipping method
		add_action('woocommerce_checkout_update_order_review', [$this, 'capture_selected_loadlink_method']);

		// Enqueue checkout assets to support WooCommerce Blocks UI
		add_action('wp_enqueue_scripts', [$this, 'enqueue_checkout_assets']);

		// AJAX endpoint to set dropoff type in session (works for guests too)
		add_action('wp_ajax_loadlink_set_dropoff_type', [$this, 'ajax_set_dropoff_type']);
		add_action('wp_ajax_nopriv_loadlink_set_dropoff_type', [$this, 'ajax_set_dropoff_type']);

		// Product: Packaging Type field under Shipping tab
		add_action('woocommerce_product_options_dimensions', [$this, 'product_packaging_type_field'], 20);

		add_action('woocommerce_admin_process_product_object', [$this, 'save_product_packaging_type']);
		add_action('woocommerce_process_product_meta', [$this, 'save_product_packaging_type_legacy']);
    }
    
    public function woocommerce_missing_notice() {
        echo '<div class="error"><p><strong>Loadlink for WooCommerce</strong> requires WooCommerce to be installed and active.</p></div>';
    }

    public function load_shipping() {
        require_once LOADLINK_PLUGIN_PATH . 'includes/class-loadlink-rates-controller.php';
        require_once LOADLINK_PLUGIN_PATH . 'includes/class-loadlink-shipping-method.php';
    }

    public function register_shipping($methods) {
        // Ensure the shipping method class is loaded
        if (!class_exists('Loadlink_Shipping_Method')) {
            $this->load_shipping();
        }
        
        // Only register if class exists
        if (class_exists('Loadlink_Shipping_Method')) {
            $methods['loadlink_rates'] = 'Loadlink_Shipping_Method';
            
            // Debug: Log that we're registering the shipping method
            if (function_exists('wc_get_logger')) {
                wc_get_logger()->info('Successfully registered Loadlink shipping method: loadlink_rates', ['source' => 'loadlink']);
            }
        } else {
            // Debug: Log error
            if (function_exists('wc_get_logger')) {
                wc_get_logger()->error('Loadlink_Shipping_Method class not found when trying to register', ['source' => 'loadlink']);
            }
        }
        
        return $methods;
    }

    public function enqueue_checkout_assets() {
		if (function_exists('is_checkout') && is_checkout()) {
			wp_enqueue_style(
				'loadlink-checkout-styles',
				LOADLINK_PLUGIN_URL . 'assets/loadlink.css',
				[],
				LOADLINK_VERSION
			);
			wp_enqueue_script(
				'loadlink-checkout-blocks',
				LOADLINK_PLUGIN_URL . 'assets/loadlink-checkout.js',
				['jquery'],
				LOADLINK_VERSION,
				true
			);
			wp_localize_script('loadlink-checkout-blocks', 'loadlinkAjax', [
				'ajaxUrl' => admin_url('admin-ajax.php'),
				'nonce' => wp_create_nonce('loadlink_set_dropoff_type'),
				'confirmNonce' => wp_create_nonce('loadlink_confirm_order'),
				'default' => get_option('loadlink_delivery_building_type', 'Residential'),
				'sessionValue' => (function_exists('WC') && WC()->session) ? WC()->session->get('loadlink_dropoff_building_type') : ''
			]);
		}
	}

	public function ajax_set_dropoff_type() {
		check_ajax_referer('loadlink_set_dropoff_type', 'nonce');
		$type = isset($_POST['value']) ? sanitize_text_field($_POST['value']) : '';
		if ($type !== 'Residential' && $type !== 'Commercial') {
			wp_send_json_error(['message' => 'Invalid value']);
		}
		if (function_exists('WC') && WC()->session) {
			WC()->session->set('loadlink_dropoff_building_type', $type);
			wp_send_json_success(['stored' => $type]);
		}
		wp_send_json_error(['message' => 'No session']);
	}

	// Add Packaging Type select to Product > Shipping tab
	public function product_packaging_type_field() {
		$post_id = get_the_ID();
		$current = $post_id ? get_post_meta($post_id, '_loadlink_packaging_code', true) : '';
		if (!function_exists('woocommerce_wp_select')) {
			// Ensure admin helpers are available
			if (function_exists('WC')) {
				$path = WC()->plugin_path() . '/includes/admin/wc-meta-box-functions.php';
				if (file_exists($path)) {
					include_once $path;
				}
			}
		}

		$choices = [
			'' => __('Select Packaging Type', 'loadlink'),
			'CR' => __('Crate', 'loadlink'),
			'DR' => __('Drum', 'loadlink'),
			'RO' => __('Roll', 'loadlink'),
			'BX' => __('Box', 'loadlink'),
			'EN' => __('Envelope', 'loadlink'),
			'SA' => __('Satchel', 'loadlink'),
			'PA' => __('Pallet', 'loadlink'),
			'BG' => __('Bag', 'loadlink'),
			'CT' => __('Carton', 'loadlink'),
			'PC' => __('Piece', 'loadlink')
		];

		if (function_exists('woocommerce_wp_select')) {
			woocommerce_wp_select([
				'id' => '_loadlink_packaging_code',
				'label' => __('Packaging Type', 'loadlink'),
				'description' => __('Used for Loadlink rate requests', 'loadlink'),
				'value' => $current,
				'options' => $choices
			]);
			return;
		}

		// Fallback render
		echo '<p class="form-field _loadlink_packaging_code_field">';
		echo '<label for="_loadlink_packaging_code">' . esc_html__('Packaging Type', 'loadlink') . '</label>';
		echo '<select id="_loadlink_packaging_code" name="_loadlink_packaging_code" class="select short">';
		foreach ($choices as $val => $label) {
			$selected = selected($current, $val, false);
			echo '<option value="' . esc_attr($val) . '" ' . $selected . '>' . esc_html($label) . '</option>';
		}
		echo '</select>';
		echo '<span class="description">' . esc_html__('Used for Loadlink rate requests', 'loadlink') . '</span>';
		echo '</p>';
	}

	// Separate meta box for Packaging Type (appears on the right sidebar)
	public function register_packaging_metabox() {
		add_meta_box(
			'loadlink_packaging_metabox',
			__('Loadlink Packaging', 'loadlink'),
			[$this, 'render_packaging_metabox'],
			'product',
			'side',
			'default'
		);
	}

	public function render_packaging_metabox($post) {
		$choices = [
			'' => __('Select Packaging Type', 'loadlink'),
			'CR' => __('Crate', 'loadlink'),
			'DR' => __('Drum', 'loadlink'),
			'RO' => __('Roll', 'loadlink'),
			'BX' => __('Box', 'loadlink'),
			'EN' => __('Envelope', 'loadlink'),
			'SA' => __('Satchel', 'loadlink'),
			'PA' => __('Pallet', 'loadlink'),
			'BG' => __('Bag', 'loadlink'),
			'CT' => __('Carton', 'loadlink'),
			'PC' => __('Piece', 'loadlink')
		];
		$current = get_post_meta($post->ID, '_loadlink_packaging_code', true);
		wp_nonce_field('loadlink_packaging_nonce', 'loadlink_packaging_nonce_field');
		echo '<p><label for="_loadlink_packaging_code"><strong>' . esc_html__('Packaging Type', 'loadlink') . '</strong></label></p>';
		echo '<p><select id="_loadlink_packaging_code" name="_loadlink_packaging_code" style="width:100%">';
		foreach ($choices as $val => $label) {
			$selected = selected($current, $val, false);
			echo '<option value="' . esc_attr($val) . '" ' . $selected . '>' . esc_html($label) . '</option>';
		}
		echo '</select></p>';
		echo '<p class="description">' . esc_html__('Used for Loadlink rate requests', 'loadlink') . '</p>';
	}

	// Save Packaging Type to product meta
	public function save_product_packaging_type($product) {
		$code = isset($_POST['_loadlink_packaging_code']) ? sanitize_text_field($_POST['_loadlink_packaging_code']) : '';
		$product->update_meta_data('_loadlink_packaging_code', $code);
	}

	// Legacy save hook compatibility
	public function save_product_packaging_type_legacy($post_id) {
		if (isset($_POST['_loadlink_packaging_code'])) {
			update_post_meta($post_id, '_loadlink_packaging_code', sanitize_text_field($_POST['_loadlink_packaging_code']));
		}
	}

	/**
	 * Add Dropoff Building Type field to checkout (under shipping fields)
	 */
	public function add_dropoff_building_type_field($fields) {
		$default = get_option('loadlink_delivery_building_type', 'Residential');
		$fields['shipping']['shipping_dropoff_building_type'] = [
			'type' => 'select',
			'label' => __('Dropoff Building Type', 'loadlink'),
			'required' => false,
			'class' => ['form-row-wide', 'update_totals_on_change'],
			'options' => [
				'Residential' => __('Residential', 'loadlink'),
				'Commercial' => __('Commercial', 'loadlink')
			],
			'default' => $default,
			'priority' => 120
		];

		// Also add to billing fields so it's visible when "Ship to different address" is unchecked
		$fields['billing']['billing_dropoff_building_type'] = [
			'type' => 'select',
			'label' => __('Dropoff Building Type', 'loadlink'),
			'required' => false,
			'class' => ['form-row-wide', 'update_totals_on_change'],
			'options' => [
				'Residential' => __('Residential', 'loadlink'),
				'Commercial' => __('Commercial', 'loadlink')
			],
			'default' => $default,
			'priority' => 120
		];

		return $fields;
	}

	/**
	 * Capture selected Dropoff Building Type into session during order review updates
	 */
	public function capture_dropoff_building_type_session($post_data) {
		parse_str($post_data, $data);
		$val = isset($data['shipping_dropoff_building_type']) ? $data['shipping_dropoff_building_type'] : '';
		
		// Fallback to billing field if shipping field is not set (e.g. ship to different address is unchecked)
		if (empty($val) && isset($data['billing_dropoff_building_type'])) {
			$val = $data['billing_dropoff_building_type'];
		}
		
		if (function_exists('WC') && WC()->session) {
			if ($val === 'Residential' || $val === 'Commercial') {
				WC()->session->set('loadlink_dropoff_building_type', $val);
			} else {
				// Reset to default if not provided
				WC()->session->set('loadlink_dropoff_building_type', get_option('loadlink_delivery_building_type', 'Residential'));
			}
		}
	}

	/**
	 * Save Dropoff Building Type to order meta
	 */
	public function save_dropoff_building_type_order_meta($order_id) {
		$val = isset($_POST['shipping_dropoff_building_type']) ? sanitize_text_field($_POST['shipping_dropoff_building_type']) : '';
		
		// Fallback to billing field
		if (empty($val) && isset($_POST['billing_dropoff_building_type'])) {
			$val = sanitize_text_field($_POST['billing_dropoff_building_type']);
		}

		if ($val !== '') {
			update_post_meta($order_id, '_loadlink_dropoff_building_type', $val);
		}
	}
	
	/**
     * Sync billing fields to shipping fields if user has not selected a different shipping address.
     * Also copies the billing phone into both shipping_phone and shipping_dropoff_phone.
     */
    public function sync_billing_to_shipping($data) {
        // Only run if customer did NOT select "Ship to a different address"
        if (!empty($data['ship_to_different_address'])) {
            return $data;
        }
    
        $fields = [
            'first_name', 'last_name', 'company',
            'address_1', 'address_2', 'city',
            'state', 'postcode', 'country'
        ];
    
        // Copy all basic address fields
        foreach ($fields as $field) {
            $billing_field  = 'billing_' . $field;
            $shipping_field = 'shipping_' . $field;
    
            if (empty($data[$shipping_field]) && !empty($data[$billing_field])) {
                $data[$shipping_field] = $data[$billing_field];
            }
        }
    
        // --- Copy phone numbers ---
        if (!empty($data['billing_phone'])) {
            $data['shipping_phone']          = $data['billing_phone'];
            $data['shipping_dropoff_phone']  = $data['billing_phone']; // for Loadlink compatibility
        }
    
        return $data;
    }


	
	/**
	 * Capture selected Loadlink shipping method and store service quote ID
	 */
	public function capture_selected_loadlink_method($post_data) {
		if (!function_exists('WC') || !WC()->session) {
			Loadlink_Logger::log('WC session not available for method capture', 'warning');
			return;
		}
		
		Loadlink_Logger::log('=== SHIPPING METHOD CAPTURE DEBUG ===', 'debug');
		Loadlink_Logger::log('Post data: ' . $post_data, 'debug');
		
		parse_str($post_data, $data);
		Loadlink_Logger::log('Parsed data: ' . json_encode($data), 'debug');
		
		if (isset($data['shipping_method'])) {
			$shipping_methods = $data['shipping_method'];
			Loadlink_Logger::log('Shipping methods found: ' . json_encode($shipping_methods), 'debug');
			
			foreach ($shipping_methods as $package_key => $method_id) {
				Loadlink_Logger::log('Processing method: ' . $method_id . ' for package: ' . $package_key, 'debug');
				
				if (strpos($method_id, 'loadlink_rates') !== false) {
					// Extract the service name from the method ID
					$service_name = str_replace('loadlink_rates:', '', $method_id);
					Loadlink_Logger::log('Extracted service name: ' . $service_name, 'debug');
					
					// Get freight data from session
					$freight_data = WC()->session->get('loadlink_freight_data');
					Loadlink_Logger::log('Freight data from session: ' . json_encode($freight_data), 'debug');
					
					if ($freight_data && isset($freight_data['service_quote_ids'][$service_name])) {
						$service_quote_id = $freight_data['service_quote_ids'][$service_name];
						
						// Store the selected service quote ID in session
						WC()->session->set('loadlink_selected_service_quote_id', $service_quote_id);
						WC()->session->set('loadlink_selected_service_name', $service_name);
						
						Loadlink_Logger::log('Captured selected Loadlink service: ' . $service_name . ' (Quote ID: ' . $service_quote_id . ')', 'info');
					} else {
						Loadlink_Logger::log('WARNING: Service quote ID not found for service: ' . $service_name, 'warning');
					}
				} else {
					Loadlink_Logger::log('Not a Loadlink method: ' . $method_id, 'debug');
				}
			}
		} else {
			Loadlink_Logger::log('No shipping_method in post data', 'debug');
		}
	}

	/**
	 * Add required Dropoff Phone field to checkout (under shipping fields)
	 */
	public function add_dropoff_phone_field($fields) {
		$fields['shipping']['shipping_dropoff_phone'] = [
			'type' => 'text',
			'label' => __('Dropoff Phone', 'loadlink'),
			'placeholder' => __('e.g. 0400 123 456', 'loadlink'),
			'required' => true,
			'priority' => 121
		];
		return $fields;
	}

	/**
	 * Server-side validation for Dropoff Phone field
	 */
	/*public function validate_dropoff_phone_field() {
		if (!isset($_POST['shipping_dropoff_phone']) || trim((string) $_POST['shipping_dropoff_phone']) === '') {
			wc_add_notice(__('Please enter a Dropoff Phone number.', 'loadlink'), 'error');
		}
	}*/
	public function validate_dropoff_phone_field() {

        // Try to read the shipping drop-off phone
        $dropoff_phone = isset($_POST['shipping_dropoff_phone'])
            ? trim((string) $_POST['shipping_dropoff_phone'])
            : '';
    
        // If empty, fall back to billing phone
        if ($dropoff_phone === '' && isset($_POST['billing_phone'])) {
            $dropoff_phone = trim((string) $_POST['billing_phone']);
        }
    
        // If still empty, show the error
        if ($dropoff_phone === '') {
            wc_add_notice(__('Please provide a drop-off phone number.', 'loadlink'), 'error');
        }
    }


	/**
	 * Save Dropoff Phone to order meta and WC session (for Blocks compatibility)
	 */
	/*public function save_dropoff_phone_order_meta($order_id) {
		if (isset($_POST['shipping_dropoff_phone'])) {
			$phone = sanitize_text_field($_POST['shipping_dropoff_phone']);
			if ($phone !== '') {
				update_post_meta($order_id, '_loadlink_dropoff_phone', $phone);
				// Also store to WC session for use in payload during next rate or order step
				if (function_exists('WC') && WC()->session) {
					WC()->session->set('loadlink_dropoff_phone', $phone);
				}
			}
		}
	}*/
	public function save_dropoff_phone_order_meta($order_id) {

        $phone = '';
    
        if (isset($_POST['shipping_dropoff_phone']) && $_POST['shipping_dropoff_phone'] !== '') {
            $phone = sanitize_text_field($_POST['shipping_dropoff_phone']);
        } elseif (isset($_POST['billing_phone']) && $_POST['billing_phone'] !== '') {
            $phone = sanitize_text_field($_POST['billing_phone']);
        }
    
        if ($phone !== '') {
            update_post_meta($order_id, '_loadlink_dropoff_phone', $phone);
    
            if (function_exists('WC') && WC()->session) {
                WC()->session->set('loadlink_dropoff_phone', $phone);
            }
        }
    }


	/**
	 * Require native shipping phone and set layout to one row (full width)
	 */
	public function require_shipping_phone_field($fields) {
		if (!isset($fields['shipping'])) {
			return $fields;
		}
		if (!isset($fields['shipping']['shipping_phone'])) {
			return $fields;
		}
		$fields['shipping']['shipping_phone']['required'] = true;
		$fields['shipping']['shipping_phone']['class'] = ['form-row-wide'];
		$fields['shipping']['shipping_phone']['priority'] = 122;
		return $fields;
	}

	/**
	 * Server-side validation for native shipping phone
	 */
	/*public function validate_native_shipping_phone() {
		if (!isset($_POST['shipping_phone']) || trim((string) $_POST['shipping_phone']) === '') {
			wc_add_notice(__('Please enter a Phone number for shipping address.', 'loadlink'), 'error');
		}
	}*/
	public function validate_native_shipping_phone() {

        error_log('Checkout POST keys: ' . implode(', ', array_keys($_POST)));

        $shipping_phone = isset($_POST['shipping_phone']) ? trim((string) $_POST['shipping_phone']) : '';
        $billing_phone  = isset($_POST['billing_phone'])  ? trim((string) $_POST['billing_phone'])  : '';
    
        if ($shipping_phone === '' && $billing_phone === '') {
            wc_add_notice(__('Please provide a phone number for shipping address.', 'loadlink'), 'error');
        } else {
            // Always sync billing → shipping phone if shipping is empty
            if ($shipping_phone === '' && $billing_phone !== '') {
                $_POST['shipping_phone'] = $billing_phone;
            }
        }
    }

}
