<?php
class Loadlink_Order_Confirmation {
    
    public static function init() {
        // Hook into WooCommerce order processing (traditional checkout)
        add_action('woocommerce_checkout_order_processed', [__CLASS__, 'save_freight_data_to_order'], 5, 1); // Run early to save data
        add_action('woocommerce_checkout_order_processed', [__CLASS__, 'process_order_confirmation'], 10, 1);
        add_action('woocommerce_order_status_processing', [__CLASS__, 'confirm_freight_on_processing'], 10, 1);
        add_action('woocommerce_order_status_completed', [__CLASS__, 'confirm_freight_on_completed'], 10, 1);
        
        // Main checkout hooks - run early to ensure data is saved
        add_action('woocommerce_checkout_order_processed', [__CLASS__, 'save_freight_data_to_order'], 5, 1);
        add_action('woocommerce_new_order', [__CLASS__, 'save_freight_data_to_order'], 5, 1);
        add_action('woocommerce_checkout_create_order', [__CLASS__, 'save_freight_data_to_order'], 5, 1);
        
        // WooCommerce Blocks checkout hooks
        add_action('woocommerce_store_api_checkout_order_processed', [__CLASS__, 'handle_blocks_checkout_order'], 10, 1);
        add_action('woocommerce_store_api_checkout_update_order_meta', [__CLASS__, 'save_freight_data_blocks'], 5, 1);
        
        // Additional hooks for better coverage
        add_action('woocommerce_new_order', [__CLASS__, 'handle_new_order'], 10, 1);
        add_action('woocommerce_payment_complete', [__CLASS__, 'handle_payment_complete'], 10, 1);
        
        // AJAX endpoints for Blocks checkout
        add_action('wp_ajax_loadlink_confirm_order', [__CLASS__, 'ajax_confirm_order']);
        add_action('wp_ajax_nopriv_loadlink_confirm_order', [__CLASS__, 'ajax_confirm_order']);
        add_action('wp_ajax_loadlink_test_freight_save', [__CLASS__, 'ajax_test_freight_save']);
        
        // Hook to capture shipping method selection during Blocks checkout
        add_action('woocommerce_store_api_checkout_update_order_from_request', [__CLASS__, 'capture_blocks_shipping_selection'], 10, 2);
        add_action('woocommerce_store_api_checkout_order_processed', [__CLASS__, 'capture_blocks_shipping_selection_final'], 5, 1);
    }
    
    /**
     * Save freight data from session to order meta during checkout
     */
    public static function save_freight_data_to_order($order_id) {
        Loadlink_Logger::log('=== SAVE FREIGHT DATA TO ORDER CALLED ===', 'info');
        Loadlink_Logger::log('Order ID: ' . $order_id, 'info');
        
        $order = wc_get_order($order_id);
        if (!$order) {
            Loadlink_Logger::log('Could not retrieve order ' . $order_id, 'error');
            return;
        }
        
        // Check if this order uses Loadlink shipping
        $shipping_methods = $order->get_shipping_methods();
        $has_loadlink = false;
        
        foreach ($shipping_methods as $shipping_method) {
            if (strpos($shipping_method->get_method_id(), 'loadlink') !== false) {
                $has_loadlink = true;
                break;
            }
        }
        
        if (!$has_loadlink) {
            Loadlink_Logger::log('Order ' . $order_id . ' does not use Loadlink shipping, skipping freight data save', 'debug');
            return;
        }
        
        Loadlink_Logger::log('Order ' . $order_id . ' uses Loadlink shipping, proceeding with freight data save', 'debug');
        
        // Get freight data from session
        $freight_data = null;
        $selected_service_quote_id = null;
        $selected_service_name = null;
        
        if (function_exists('WC') && WC()->session) {
            $freight_data = WC()->session->get('loadlink_freight_data');
            $selected_service_quote_id = WC()->session->get('loadlink_selected_service_quote_id');
            $selected_service_name = WC()->session->get('loadlink_selected_service_name');
            
            Loadlink_Logger::log('Session freight data: ' . json_encode($freight_data), 'debug');
            Loadlink_Logger::log('Selected service quote ID: ' . $selected_service_quote_id, 'debug');
            Loadlink_Logger::log('Selected service name: ' . $selected_service_name, 'debug');
        }
        
        // If no session data, try to get from API or create a basic freight record
        if (!$freight_data || !isset($freight_data['freight_id'])) {
            Loadlink_Logger::log('No freight data in session, attempting to create basic freight record', 'debug');
            
            // Try to get freight data from API if available
            if (class_exists('Loadlink_Client')) {
                $client = new Loadlink_Client();
                $api_freight_data = $client->get_all_freight_data();
                
                if ($api_freight_data && isset($api_freight_data['freight_id'])) {
                    $freight_data = $api_freight_data;
                    Loadlink_Logger::log('Retrieved freight data from API: ' . json_encode($freight_data), 'debug');
                }
            }
            
            // If still no freight data, create a basic record
            if (!$freight_data || !isset($freight_data['freight_id'])) {
                $freight_data = [
                    'freight_id' => 'temp_' . $order_id . '_' . time(),
                    'name' => 'LoadLink Dev',
                    'pickup_address' => [
                        'name' => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
                        'address' => $order->get_billing_address_1(),
                        'suburb' => $order->get_billing_city(),
                        'postcode' => $order->get_billing_postcode(),
                        'state' => $order->get_billing_state()
                    ],
                    'delivery_address' => [
                        'name' => $order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name(),
                        'address' => $order->get_shipping_address_1(),
                        'suburb' => $order->get_shipping_city(),
                        'postcode' => $order->get_shipping_postcode(),
                        'state' => $order->get_shipping_state()
                    ],
                    'service_quote_ids' => [],
                    'selected_service_quote_id' => null,
                    'selected_service_name' => null
                ];
                Loadlink_Logger::log('Created basic freight data: ' . json_encode($freight_data), 'debug');
            }
        }
        
        if ($freight_data && isset($freight_data['freight_id'])) {
            // Save freight ID
            $order->update_meta_data('_loadlink_freight_id', $freight_data['freight_id']);
            Loadlink_Logger::log('Saved freight_id ' . $freight_data['freight_id'] . ' to order ' . $order_id, 'debug');
            
            // Save service quote ID (prefer selected one, fallback to first available)
            $service_quote_id = null;
            if ($selected_service_quote_id) {
                $service_quote_id = $selected_service_quote_id;
                Loadlink_Logger::log('Using selected service quote ID: ' . $service_quote_id . ' for service: ' . $selected_service_name, 'info');
            } elseif (isset($freight_data['service_quote_ids']) && !empty($freight_data['service_quote_ids'])) {
                // Use the first available service quote ID
                $service_quote_id = reset($freight_data['service_quote_ids']);
                Loadlink_Logger::log('WARNING: Using first available service quote ID (no selection made): ' . $service_quote_id, 'warning');
            }
            
            if ($service_quote_id) {
                $order->update_meta_data('_loadlink_service_quote_id', $service_quote_id);
                Loadlink_Logger::log('Saved service_quote_id ' . $service_quote_id . ' to order ' . $order_id, 'info');
                
                // Also store the selected service quote ID in the freight data for reference
                $freight_data['selected_service_quote_id'] = $service_quote_id;
                $freight_data['selected_service_name'] = $selected_service_name;
            }
            
            // Save all freight data for reference
            $order->update_meta_data('_loadlink_freight_data', $freight_data);
            
            // Save selected service name if available
            if ($selected_service_name) {
                $order->update_meta_data('_loadlink_selected_service_name', $selected_service_name);
            }
            
            $order->save();
            
            Loadlink_Logger::log('Successfully saved Loadlink freight data to order ' . $order_id, 'info');
        } else {
            Loadlink_Logger::log('No freight data found in session for order ' . $order_id, 'warning');
        }
    }
    
    public static function process_order_confirmation($order_id) {
        Loadlink_Logger::log('process_order_confirmation called for order: ' . $order_id, 'debug');
        
        $order = wc_get_order($order_id);
        if (!$order) {
            Loadlink_Logger::log('Could not retrieve order ' . $order_id . ' for confirmation', 'error');
            return;
        }
        
        // Block duplicate confirmations
        $confirmation_response = $order->get_meta('_loadlink_confirmation_response', true);
        if (!empty($confirmation_response)) {
            // Loadlink_Logger::log('Order ' . $order_id . ' already confirmed. Skipping duplicate confirmation trigger.', 'info');
            // $order->add_order_note('Loadlink: Freight already confirmed. Skipped duplicate confirmation.');
            return;
        }
        
        // Check if this order uses Loadlink shipping
        $shipping_methods = $order->get_shipping_methods();
        $has_loadlink = false;
        
        foreach ($shipping_methods as $shipping_method) {
            if (strpos($shipping_method->get_method_id(), 'loadlink') !== false) {
                $has_loadlink = true;
                break;
            }
        }
        
        if (!$has_loadlink) {
            Loadlink_Logger::log('Order ' . $order_id . ' does not use Loadlink shipping, skipping confirmation', 'debug');
            return;
        }
        
        // Get service quote ID from order meta
        $service_quote_id = self::get_service_quote_id_from_order($order);
        
        if (!$service_quote_id) {
            Loadlink_Logger::log('Order ' . $order_id . ' missing service_quote_id, skipping confirmation', 'warning');
            return;
        }
        
        // Prepare order data for confirmation
        $order_data = self::prepare_order_data($order);
        
        // Confirm the freight order
        self::confirm_freight_order($service_quote_id, $order_data, $order);
    }
    
    public static function confirm_freight_on_processing($order_id) {
        // Alternative hook for when order status changes to processing
        self::process_order_confirmation($order_id);
    }
    
    public static function confirm_freight_on_completed($order_id) {
        // Alternative hook for when order status changes to completed
        self::process_order_confirmation($order_id);
    }
    
    private static function get_freight_id_from_order($order) {
        // Try to get from order meta first
        $freight_id = $order->get_meta('_loadlink_freight_id');
        if ($freight_id) {
            Loadlink_Logger::log('Found freight_id in order meta: ' . $freight_id, 'debug');
            return $freight_id;
        }
        
        // Try to get from freight data in order meta
        $freight_data = $order->get_meta('_loadlink_freight_data');
        if ($freight_data && isset($freight_data['freight_id'])) {
            Loadlink_Logger::log('Found freight_id in order freight data: ' . $freight_data['freight_id'], 'debug');
            return $freight_data['freight_id'];
        }
        
        // Try to get from session data
        if (function_exists('WC') && WC()->session) {
            $freight_data = WC()->session->get('loadlink_freight_data');
            if (isset($freight_data['freight_id'])) {
                Loadlink_Logger::log('Found freight_id in session: ' . $freight_data['freight_id'], 'debug');
                return $freight_data['freight_id'];
            }
        }
        
        // Try to get from static freight data
        if (class_exists('Loadlink_Client')) {
            $freight_id = Loadlink_Client::get_freight_id();
            if ($freight_id) {
                Loadlink_Logger::log('Found freight_id in static data: ' . $freight_id, 'debug');
                return $freight_id;
            }
        }
        
        Loadlink_Logger::log('No freight_id found for order ' . $order->get_id(), 'warning');
        return null;
    }
    
    private static function get_service_quote_id_from_order($order) {
        Loadlink_Logger::log('=== SERVICE QUOTE ID RETRIEVAL DEBUG ===', 'debug');
        Loadlink_Logger::log('Order ID: ' . $order->get_id(), 'debug');
        
        // Try to get from order meta first
        $service_quote_id = $order->get_meta('_loadlink_service_quote_id');
        if ($service_quote_id) {
            Loadlink_Logger::log('Found service_quote_id in order meta: ' . $service_quote_id, 'info');
            return $service_quote_id;
        }
        Loadlink_Logger::log('No service_quote_id in order meta', 'debug');
        
        // Try to get from freight data in order meta
        $freight_data = $order->get_meta('_loadlink_freight_data');
        if ($freight_data && isset($freight_data['service_quote_ids']) && !empty($freight_data['service_quote_ids'])) {
            Loadlink_Logger::log('Available service_quote_ids in order freight data: ' . json_encode($freight_data['service_quote_ids']), 'debug');
            
            // Check if there's a selected service quote ID in the freight data
            if (isset($freight_data['selected_service_quote_id'])) {
                $service_quote_id = $freight_data['selected_service_quote_id'];
                Loadlink_Logger::log('Using selected service_quote_id from freight data: ' . $service_quote_id, 'info');
                return $service_quote_id;
            }
            
            // Fallback to first available (this is the problem!)
            $service_quote_id = reset($freight_data['service_quote_ids']);
            Loadlink_Logger::log('WARNING: Using first available service_quote_id (fallback): ' . $service_quote_id, 'warning');
            return $service_quote_id;
        }
        Loadlink_Logger::log('No freight data or service_quote_ids in order meta', 'debug');
        
        // Try to get from session data (selected service quote ID)
        if (function_exists('WC') && WC()->session) {
            $selected_service_quote_id = WC()->session->get('loadlink_selected_service_quote_id');
            if ($selected_service_quote_id) {
                Loadlink_Logger::log('Found selected service_quote_id in session: ' . $selected_service_quote_id, 'info');
                return $selected_service_quote_id;
            }
            Loadlink_Logger::log('No selected service_quote_id in session', 'debug');
            
            // Fallback to freight data in session
            $freight_data = WC()->session->get('loadlink_freight_data');
            if (isset($freight_data['service_quote_ids']) && !empty($freight_data['service_quote_ids'])) {
                Loadlink_Logger::log('Available service_quote_ids in session freight data: ' . json_encode($freight_data['service_quote_ids']), 'debug');
                $service_quote_id = reset($freight_data['service_quote_ids']);
                Loadlink_Logger::log('WARNING: Using first available service_quote_id from session (fallback): ' . $service_quote_id, 'warning');
                return $service_quote_id;
            }
        }
        
        Loadlink_Logger::log('No service_quote_id found for order ' . $order->get_id(), 'error');
        return null;
    }
    
    private static function prepare_order_data($order) {
        // Get pickup address (store address)
        $pickup_address = self::get_pickup_address();
        
        // Get dropoff address (shipping address)
        $dropoff_address = self::get_dropoff_address($order);
        
        return [
            'order' => $order,
            'pickup_address' => $pickup_address,
            'dropoff_address' => $dropoff_address
        ];
    }
    
    private static function get_pickup_address() {
        // Get store address from WooCommerce settings
        $store_address = get_option('woocommerce_store_address', '');
        $store_address_2 = get_option('woocommerce_store_address_2', '');
        $store_city = get_option('woocommerce_store_city', '');
        $store_postcode = get_option('woocommerce_store_postcode', '');
        $store_country = get_option('woocommerce_default_country', '');
        $store_state = '';
        
        // Extract state from country code (e.g., AU:QLD)
        if (strpos($store_country, ':') !== false) {
            list($country, $state) = explode(':', $store_country);
        }
        
        // Derive phone parts from a single store phone field
        $store_phone = get_option('woocommerce_store_phone', '');
        $parsed = self::parse_phone_number($store_phone, 'AU');
        $store_phone_area = $parsed['area_code'];
        $store_phone = $parsed['phone'];

        return [
            'state' => $state,
            'suburb' => $store_city,
            'postcode' => $store_postcode,
            'address1' => $store_address,
            'address2' => $store_address_2,
            'address3' => '',
            'company_name' => get_option('woocommerce_store_name', ''),
            'contact_name' => get_option('woocommerce_store_contact_name', 'Store Manager'),
            'email' => get_option('woocommerce_store_email', get_option('admin_email')),
            'phone_area_code' => $store_phone_area,
            'phone' => $store_phone
        ];
    }
    
    private static function get_dropoff_address($order) {
        $shipping = $order->get_address('shipping');
        $billing = $order->get_address('billing');

        // Helper: prefer shipping value; fallback to billing
        $pick = function($key) use ($shipping, $billing) {
            $s = isset($shipping[$key]) ? trim((string) $shipping[$key]) : '';
            if ($s !== '') {
                return $s;
            }
            return isset($billing[$key]) ? trim((string) $billing[$key]) : '';
        };

        // Phone priority for payload: shipping phone if present, otherwise billing phone
        $shipping_phone = method_exists($order, 'get_shipping_phone') ? $order->get_shipping_phone() : '';
        $raw_phone = trim((string) $shipping_phone) !== '' ? $shipping_phone : $order->get_billing_phone();
        $parsed_phone = self::parse_phone_number($raw_phone, 'AU');

        // Names: prefer shipping name, else billing
        $first_name = $pick('first_name');
        $last_name = $pick('last_name');
        $contact_name = trim($first_name . ' ' . $last_name);

        return [
            'state' => $pick('state'),
            'suburb' => $pick('city'),
            'postcode' => $pick('postcode'),
            'address1' => $pick('address_1'),
            'address2' => $pick('address_2'),
            'address3' => '',
            'company_name' => $pick('company'),
            'contact_name' => $contact_name,
            // WooCommerce has only billing email field; use it
            'email' => $order->get_billing_email(),
            'phone_area_code' => $parsed_phone['area_code'],
            'phone' => $parsed_phone['phone']
        ];
    }

    /**
     * Parse a phone number into area code (country code) and local phone part.
     * Currently optimized for AU numbers.
     */
    public static function parse_phone_number($raw_phone, $default_country = 'AU') {
        $raw = trim((string) $raw_phone);
        if ($raw === '') {
            return ['area_code' => '', 'phone' => ''];
        }

        // Normalize: keep digits and leading +
        $normalized = preg_replace('/[^\d+]/', '', $raw);

        // AU handling
        if (strtoupper($default_country) === 'AU') {
            // If starts with +61
            if (strpos($normalized, '+61') === 0) {
                $rest = substr($normalized, 3);
                // Remove leading 0 if present after country code
                if (strpos($rest, '0') === 0) {
                    $rest = substr($rest, 1);
                }
                return ['area_code' => '61', 'phone' => $rest];
            }
            // If starts with 61 without +
            if (strpos($normalized, '61') === 0) {
                $rest = substr($normalized, 2);
                if (strpos($rest, '0') === 0) {
                    $rest = substr($rest, 1);
                }
                return ['area_code' => '61', 'phone' => $rest];
            }
            // If local leading 0 (e.g., 04xxxxxxx or 0x...)
            if (strpos($normalized, '0') === 0) {
                return ['area_code' => '61', 'phone' => substr($normalized, 1)];
            }
            // Fallback: assume AU, try to use as-is
            return ['area_code' => '61', 'phone' => $normalized];
        }

        // Generic fallback: if starts with +, take up to 3-digit country code
        if (strpos($normalized, '+') === 0) {
            if (preg_match('/^\+(\d{1,3})(\d+)$/', $normalized, $m)) {
                return ['area_code' => $m[1], 'phone' => $m[2]];
            }
        }

        // Unknown format: return digits as phone, empty area code
        $digits = preg_replace('/\D/', '', $normalized);
        return ['area_code' => '', 'phone' => $digits];
    }
    
    private static function confirm_freight_order($service_quote_id, $order_data, $order) {
        try {
            // Get freight_id from order
            $freight_id = self::get_freight_id_from_order($order);
            if (!$freight_id) {
                Loadlink_Logger::log('Order ' . $order->get_id() . ' missing freight_id, cannot confirm freight', 'error');
                $order->add_order_note('Loadlink freight confirmation failed: Missing freight_id');
                return false;
            }
            
            $client = new Loadlink_Client();
            $response = $client->confirm_freight_order($service_quote_id, $order_data, $freight_id);
            
            if (is_wp_error($response)) {
                Loadlink_Logger::log('Freight confirmation failed for order ' . $order->get_id() . ': ' . $response->get_error_message(), 'error');
                
                // Add order note about the failure
                $order->add_order_note('Loadlink freight confirmation failed: ' . $response->get_error_message());
                
                return false;
            }
            
            // Parse the response
            $confirmation_data = json_decode($response, true);
            
            if ($confirmation_data) {
                // Store confirmation data in order meta
                $order->update_meta_data('_loadlink_confirmation_response', $response);
                $order->update_meta_data('_loadlink_confirmation_status', 'confirmed');
                
                if (isset($confirmation_data['confirmation_id'])) {
                    $order->update_meta_data('_loadlink_confirmation_id', $confirmation_data['confirmation_id']);
                }
                
                $order->save();
                
                // Add order note
                $order->add_order_note('Loadlink freight order confirmed successfully. Service Quote ID: ' . $service_quote_id);
                
                Loadlink_Logger::log('Freight confirmation successful for order ' . $order->get_id(), 'info');
                
                return true;
            } else {
                Loadlink_Logger::log('Failed to parse freight confirmation response for order ' . $order->get_id(), 'error');
                $order->add_order_note('Loadlink freight confirmation failed: Invalid response format');
                return false;
            }
            
        } catch (Exception $e) {
            Loadlink_Logger::log('Exception during freight confirmation for order ' . $order->get_id() . ': ' . $e->getMessage(), 'error');
            $order->add_order_note('Loadlink freight confirmation failed: ' . $e->getMessage());
            return false;
        }
    }
    
    public static function get_confirmation_status($order_id) {
        $order = wc_get_order($order_id);
        if (!$order) {
            return null;
        }
        
        return $order->get_meta('_loadlink_confirmation_status');
    }
    
    public static function get_confirmation_response($order_id) {
        $order = wc_get_order($order_id);
        if (!$order) {
            return null;
        }
        
        return $order->get_meta('_loadlink_confirmation_response');
    }
    
    /**
     * Manual test function to trigger order confirmation
     * This can be called from admin or via AJAX for testing
     */
    public static function test_order_confirmation($order_id) {
        Loadlink_Logger::log('Manual test of order confirmation for order: ' . $order_id, 'info');
        
        $order = wc_get_order($order_id);
        if (!$order) {
            Loadlink_Logger::log('Test failed: Could not retrieve order ' . $order_id, 'error');
            return false;
        }
        
        // Check if order uses Loadlink shipping
        $shipping_methods = $order->get_shipping_methods();
        $has_loadlink = false;
        
        foreach ($shipping_methods as $shipping_method) {
            if (strpos($shipping_method->get_method_id(), 'loadlink') !== false) {
                $has_loadlink = true;
                break;
            }
        }
        
        if (!$has_loadlink) {
            Loadlink_Logger::log('Test failed: Order ' . $order_id . ' does not use Loadlink shipping', 'error');
            return false;
        }
        
        // Try to process confirmation
        self::process_order_confirmation($order_id);
        
        Loadlink_Logger::log('Manual test completed for order: ' . $order_id, 'info');
        return true;
    }
    
    /**
     * Handle Blocks checkout order processing
     */
    public static function handle_blocks_checkout_order($order) {
        Loadlink_Logger::log('Blocks checkout order processed: ' . $order->get_id(), 'debug');
        
        // Save freight data first
        self::save_freight_data_blocks($order->get_id());
        
        // Process confirmation
        self::process_order_confirmation($order->get_id());
    }
    
    /**
     * Save freight data for Blocks checkout
     */
    public static function save_freight_data_blocks($order_id) {
        Loadlink_Logger::log('save_freight_data_blocks called for order: ' . $order_id, 'debug');
        
        $order = wc_get_order($order_id);
        if (!$order) {
            Loadlink_Logger::log('Could not retrieve order ' . $order_id . ' for Blocks checkout', 'error');
            return;
        }
        
        // Check if this order uses Loadlink shipping
        $shipping_methods = $order->get_shipping_methods();
        $has_loadlink = false;
        $selected_method_id = null;
        
        Loadlink_Logger::log('=== BLOCKS CHECKOUT SHIPPING METHODS DEBUG ===', 'debug');
        Loadlink_Logger::log('Order ID: ' . $order_id, 'debug');
        Loadlink_Logger::log('Shipping methods count: ' . count($shipping_methods), 'debug');
        
        foreach ($shipping_methods as $shipping_method) {
            $method_id = $shipping_method->get_method_id();
            $method_title = $shipping_method->get_method_title();
            $method_instance_id = $shipping_method->get_instance_id();
            
            Loadlink_Logger::log('Shipping method details:', 'debug');
            Loadlink_Logger::log('  - Method ID: ' . $method_id, 'debug');
            Loadlink_Logger::log('  - Method Title: ' . $method_title, 'debug');
            Loadlink_Logger::log('  - Instance ID: ' . $method_instance_id, 'debug');
            Loadlink_Logger::log('  - Full method data: ' . json_encode($shipping_method->get_data()), 'debug');
            
            if (strpos($method_id, 'loadlink') !== false) {
                $has_loadlink = true;
                $selected_method_id = $method_id;
                Loadlink_Logger::log('Found Loadlink method: ' . $method_id, 'info');
            }
        }
        
        if (!$has_loadlink) {
            Loadlink_Logger::log('Order ' . $order_id . ' does not use Loadlink shipping (Blocks checkout), skipping freight data save', 'debug');
            return;
        }
        
        Loadlink_Logger::log('Order ' . $order_id . ' uses Loadlink shipping (Blocks checkout), proceeding with freight data save', 'debug');
        
        // Get freight data from session or try alternative methods
        $freight_data = null;
        $selected_service_quote_id = null;
        $selected_service_name = null;
        
        // Try session first
        if (function_exists('WC') && WC()->session) {
            $freight_data = WC()->session->get('loadlink_freight_data');
            $selected_service_quote_id = WC()->session->get('loadlink_selected_service_quote_id');
            $selected_service_name = WC()->session->get('loadlink_selected_service_name');
        }
        
        // If we don't have selection from session, try to extract from the selected method ID
        if (!$selected_service_quote_id && $selected_method_id) {
            $extracted_service_name = null;
            $extracted_quote_id = null;
            
            // First, try to get quote_id directly from meta_data
            if (!empty($shipping_methods)) {
                foreach ($shipping_methods as $shipping_method) {
                    if (strpos($shipping_method->get_method_id(), 'loadlink') !== false) {
                        $meta_data = $shipping_method->get_meta_data();
                        
                        foreach ($meta_data as $meta) {
                            if ($meta->key === 'quote_id') {
                                $extracted_quote_id = $meta->value;
                                Loadlink_Logger::log('Found quote_id in meta_data: ' . $extracted_quote_id, 'info');
                                break;
                            }
                        }
                        
                        if ($extracted_quote_id) {
                            // Find the corresponding service name
                            if ($freight_data && isset($freight_data['service_quote_ids'])) {
                                foreach ($freight_data['service_quote_ids'] as $service => $service_quote_id) {
                                    if ($service_quote_id == $extracted_quote_id) {
                                        $extracted_service_name = $service;
                                        Loadlink_Logger::log('Found matching service for quote_id ' . $extracted_quote_id . ': ' . $extracted_service_name, 'info');
                                        break;
                                    }
                                }
                            }
                        }
                        
                        if ($extracted_service_name) {
                            break;
                        }
                    }
                }
            }
            
            // Fallback: try to extract from method title
            if (!$extracted_service_name && !empty($shipping_methods)) {
                foreach ($shipping_methods as $shipping_method) {
                    $method_title = $shipping_method->get_method_title();
                    if (strpos($method_title, 'Road Express') !== false) {
                        $extracted_service_name = 'road express';
                        break;
                    } elseif (strpos($method_title, 'Overnight Express') !== false) {
                        $extracted_service_name = 'overnight express';
                        break;
                    } elseif (strpos($method_title, 'Technology Express') !== false) {
                        $extracted_service_name = 'technology express';
                        break;
                    } elseif (strpos($method_title, 'Air Express') !== false) {
                        $extracted_service_name = 'air express';
                        break;
                    } elseif (strpos($method_title, 'ASAP Hot Shot') !== false) {
                        $extracted_service_name = 'asap hot shot';
                        break;
                    }
                }
            }
            
            if ($extracted_service_name) {
                Loadlink_Logger::log('Extracted service name from method ID/title: ' . $extracted_service_name, 'info');
                
                if ($freight_data && isset($freight_data['service_quote_ids'][$extracted_service_name])) {
                    $selected_service_quote_id = $freight_data['service_quote_ids'][$extracted_service_name];
                    $selected_service_name = $extracted_service_name;
                    Loadlink_Logger::log('Found service quote ID for ' . $extracted_service_name . ': ' . $selected_service_quote_id, 'info');
                } else {
                    Loadlink_Logger::log('WARNING: Service quote ID not found for extracted service: ' . $extracted_service_name, 'warning');
                    Loadlink_Logger::log('Available services: ' . json_encode(array_keys($freight_data['service_quote_ids'] ?? [])), 'debug');
                }
            } else {
                Loadlink_Logger::log('WARNING: Could not extract service name from method ID: ' . $selected_method_id, 'warning');
            }
        }
        
        // If no session data, try to get from API or static data
        if (!$freight_data && class_exists('Loadlink_Client')) {
            $freight_data = Loadlink_Client::get_all_freight_data();
            Loadlink_Logger::log('Retrieved freight data from API for Blocks checkout', 'debug');
        }
        
        if ($freight_data && isset($freight_data['freight_id'])) {
            // Save freight ID
            $order->update_meta_data('_loadlink_freight_id', $freight_data['freight_id']);
            Loadlink_Logger::log('Saved freight_id ' . $freight_data['freight_id'] . ' to order ' . $order_id . ' (Blocks checkout)', 'debug');
            
            // Save service quote ID (prefer selected one, fallback to first available)
            $service_quote_id = null;
            if ($selected_service_quote_id) {
                $service_quote_id = $selected_service_quote_id;
                Loadlink_Logger::log('Using selected service quote ID: ' . $service_quote_id . ' for service: ' . $selected_service_name . ' (Blocks checkout)', 'info');
            } elseif (isset($freight_data['service_quote_ids']) && !empty($freight_data['service_quote_ids'])) {
                // Use the first available service quote ID
                $service_quote_id = reset($freight_data['service_quote_ids']);
                Loadlink_Logger::log('WARNING: Using first available service quote ID (no selection made): ' . $service_quote_id . ' (Blocks checkout)', 'warning');
            }
            
            if ($service_quote_id) {
                $order->update_meta_data('_loadlink_service_quote_id', $service_quote_id);
                Loadlink_Logger::log('Saved service_quote_id ' . $service_quote_id . ' to order ' . $order_id . ' (Blocks checkout)', 'info');
                
                // Also store the selected service quote ID in the freight data for reference
                $freight_data['selected_service_quote_id'] = $service_quote_id;
                $freight_data['selected_service_name'] = $selected_service_name;
            }
            
            // Save all freight data for reference
            $order->update_meta_data('_loadlink_freight_data', $freight_data);
            
            // Save selected service name if available
            if ($selected_service_name) {
                $order->update_meta_data('_loadlink_selected_service_name', $selected_service_name);
            }
            
            // Mark as Blocks checkout
            $order->update_meta_data('_loadlink_blocks_checkout', 'yes');
            
            $order->save();
            
            Loadlink_Logger::log('Successfully saved Loadlink freight data to order ' . $order_id . ' (Blocks checkout)', 'info');
        } else {
            Loadlink_Logger::log('No freight data found for Blocks checkout order ' . $order_id, 'warning');
        }
    }
    
    /**
     * Handle new order creation
     */
    public static function handle_new_order($order_id) {
        Loadlink_Logger::log('New order created: ' . $order_id, 'debug');
        
        // Check if this is a Blocks checkout order
        $order = wc_get_order($order_id);
        if ($order && $order->get_meta('_loadlink_blocks_checkout')) {
            Loadlink_Logger::log('Processing Blocks checkout order: ' . $order_id, 'debug');
            self::process_order_confirmation($order_id);
        }
    }
    
    /**
     * Handle payment completion
     */
    public static function handle_payment_complete($order_id) {
        Loadlink_Logger::log('Payment completed for order: ' . $order_id, 'debug');
        
        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }
        
        // Check if this order uses Loadlink shipping
        $shipping_methods = $order->get_shipping_methods();
        $has_loadlink = false;
        
        foreach ($shipping_methods as $shipping_method) {
            if (strpos($shipping_method->get_method_id(), 'loadlink') !== false) {
                $has_loadlink = true;
                break;
            }
        }
        
        if ($has_loadlink) {
            Loadlink_Logger::log('Processing order confirmation for payment completed order: ' . $order_id, 'debug');
            self::process_order_confirmation($order_id);
        }
    }
    
    /**
     * Capture shipping method selection during Blocks checkout
     */
    public static function capture_blocks_shipping_selection($order, $request) {
        Loadlink_Logger::log('=== BLOCKS SHIPPING SELECTION CAPTURE ===', 'debug');
        Loadlink_Logger::log('Order ID: ' . $order->get_id(), 'debug');
        
        // Get shipping methods from the request
        $shipping_methods = $request->get_param('shipping_methods') ?? [];
        Loadlink_Logger::log('Shipping methods from request: ' . json_encode($shipping_methods), 'debug');
        
        // Also check the order's shipping methods directly
        $order_shipping_methods = $order->get_shipping_methods();
        Loadlink_Logger::log('Order shipping methods: ' . json_encode($order_shipping_methods), 'debug');
        
        foreach ($order_shipping_methods as $shipping_method) {
            $method_id = $shipping_method->get_method_id();
            $method_title = $shipping_method->get_method_title();
            
            Loadlink_Logger::log('Processing order shipping method: ' . $method_id . ' - ' . $method_title, 'debug');
            
            if (strpos($method_id, 'loadlink') !== false) {
                // First, try to get quote_id directly from meta_data
                $meta_data = $shipping_method->get_meta_data();
                $quote_id = null;
                $service_name = null;
                
                foreach ($meta_data as $meta) {
                    if ($meta->key === 'quote_id') {
                        $quote_id = $meta->value;
                        Loadlink_Logger::log('Found quote_id in meta_data: ' . $quote_id, 'info');
                        break;
                    }
                }
                
                // If we have quote_id, find the corresponding service name
                if ($quote_id) {
                    if (function_exists('WC') && WC()->session) {
                        $freight_data = WC()->session->get('loadlink_freight_data');
                        
                        if ($freight_data && isset($freight_data['service_quote_ids'])) {
                            foreach ($freight_data['service_quote_ids'] as $service => $service_quote_id) {
                                if ($service_quote_id == $quote_id) {
                                    $service_name = $service;
                                    Loadlink_Logger::log('Found matching service for quote_id ' . $quote_id . ': ' . $service_name, 'info');
                                    break;
                                }
                            }
                        }
                    }
                }
                
                // Fallback: try to extract service name from method title
                if (!$service_name) {
                    if (strpos($method_title, 'Road Express') !== false) {
                        $service_name = 'road express';
                    } elseif (strpos($method_title, 'Overnight Express') !== false) {
                        $service_name = 'overnight express';
                    } elseif (strpos($method_title, 'Technology Express') !== false) {
                        $service_name = 'technology express';
                    } elseif (strpos($method_title, 'Air Express') !== false) {
                        $service_name = 'air express';
                    } elseif (strpos($method_title, 'ASAP Hot Shot') !== false) {
                        $service_name = 'asap hot shot';
                    }
                    
                    if ($service_name) {
                        Loadlink_Logger::log('Extracted service name from method title: ' . $service_name, 'info');
                    }
                }
                
                if ($service_name) {
                    // Get freight data from session
                    if (function_exists('WC') && WC()->session) {
                        $freight_data = WC()->session->get('loadlink_freight_data');
                        
                        if ($freight_data && isset($freight_data['service_quote_ids'][$service_name])) {
                            $service_quote_id = $freight_data['service_quote_ids'][$service_name];
                            
                            // Store the selected service quote ID in session
                            WC()->session->set('loadlink_selected_service_quote_id', $service_quote_id);
                            WC()->session->set('loadlink_selected_service_name', $service_name);
                            
                            Loadlink_Logger::log('Captured Blocks checkout selection: ' . $service_name . ' (Quote ID: ' . $service_quote_id . ')', 'info');
                        } else {
                            Loadlink_Logger::log('WARNING: Service quote ID not found for Blocks checkout service: ' . $service_name, 'warning');
                            Loadlink_Logger::log('Available services: ' . json_encode(array_keys($freight_data['service_quote_ids'] ?? [])), 'debug');
                        }
                    }
                } else {
                    Loadlink_Logger::log('WARNING: Could not extract service name from method: ' . $method_id . ' - ' . $method_title, 'warning');
                }
            }
        }
    }
    
    /**
     * Final capture of Blocks checkout shipping selection after order is processed
     */
    public static function capture_blocks_shipping_selection_final($order) {
        Loadlink_Logger::log('=== BLOCKS FINAL SHIPPING SELECTION CAPTURE ===', 'debug');
        Loadlink_Logger::log('Order ID: ' . $order->get_id(), 'debug');
        
        // Get shipping methods from the order
        $shipping_methods = $order->get_shipping_methods();
        Loadlink_Logger::log('Final order shipping methods count: ' . count($shipping_methods), 'debug');
        
        foreach ($shipping_methods as $shipping_method) {
            $method_id = $shipping_method->get_method_id();
            $method_title = $shipping_method->get_method_title();
            
            Loadlink_Logger::log('Final processing shipping method: ' . $method_id . ' - ' . $method_title, 'debug');
            
            if (strpos($method_id, 'loadlink') !== false) {
                // First, try to get quote_id directly from meta_data
                $meta_data = $shipping_method->get_meta_data();
                $quote_id = null;
                $service_name = null;
                
                foreach ($meta_data as $meta) {
                    if ($meta->key === 'quote_id') {
                        $quote_id = $meta->value;
                        Loadlink_Logger::log('Final found quote_id in meta_data: ' . $quote_id, 'info');
                        break;
                    }
                }
                
                // If we have quote_id, find the corresponding service name
                if ($quote_id) {
                    if (function_exists('WC') && WC()->session) {
                        $freight_data = WC()->session->get('loadlink_freight_data');
                        
                        if ($freight_data && isset($freight_data['service_quote_ids'])) {
                            foreach ($freight_data['service_quote_ids'] as $service => $service_quote_id) {
                                if ($service_quote_id == $quote_id) {
                                    $service_name = $service;
                                    Loadlink_Logger::log('Final found matching service for quote_id ' . $quote_id . ': ' . $service_name, 'info');
                                    break;
                                }
                            }
                        }
                    }
                }
                
                // Fallback: try to extract service name from method title
                if (!$service_name) {
                    if (strpos($method_title, 'Road Express') !== false) {
                        $service_name = 'road express';
                    } elseif (strpos($method_title, 'Overnight Express') !== false) {
                        $service_name = 'overnight express';
                    } elseif (strpos($method_title, 'Technology Express') !== false) {
                        $service_name = 'technology express';
                    } elseif (strpos($method_title, 'Air Express') !== false) {
                        $service_name = 'air express';
                    } elseif (strpos($method_title, 'ASAP Hot Shot') !== false) {
                        $service_name = 'asap hot shot';
                    }
                    
                    if ($service_name) {
                        Loadlink_Logger::log('Final extracted service name from method title: ' . $service_name, 'info');
                    }
                }
                
                if ($service_name) {
                    // Get freight data from session
                    if (function_exists('WC') && WC()->session) {
                        $freight_data = WC()->session->get('loadlink_freight_data');
                        
                        if ($freight_data && isset($freight_data['service_quote_ids'][$service_name])) {
                            $service_quote_id = $freight_data['service_quote_ids'][$service_name];
                            
                            // Store the selected service quote ID in session
                            WC()->session->set('loadlink_selected_service_quote_id', $service_quote_id);
                            WC()->session->set('loadlink_selected_service_name', $service_name);
                            
                            Loadlink_Logger::log('Final captured Blocks checkout selection: ' . $service_name . ' (Quote ID: ' . $service_quote_id . ')', 'info');
                        } else {
                            Loadlink_Logger::log('WARNING: Service quote ID not found in final capture for service: ' . $service_name, 'warning');
                            Loadlink_Logger::log('Available services: ' . json_encode(array_keys($freight_data['service_quote_ids'] ?? [])), 'debug');
                        }
                    }
                } else {
                    Loadlink_Logger::log('WARNING: Could not extract service name in final capture from method: ' . $method_id . ' - ' . $method_title, 'warning');
                }
            }
        }
    }
    
    /**
     * AJAX endpoint for Blocks checkout order confirmation
     */
    public static function ajax_confirm_order() {
        Loadlink_Logger::log('=== AJAX ORDER CONFIRMATION REQUEST ===', 'info');
        Loadlink_Logger::log('POST data: ' . json_encode($_POST), 'debug');
        
        // Check if nonce is provided
        if (!isset($_POST['nonce'])) {
            Loadlink_Logger::log('ERROR: No nonce provided in AJAX request', 'error');
            wp_send_json_error(['message' => 'No nonce provided']);
        }
        
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'loadlink_confirm_order')) {
            Loadlink_Logger::log('ERROR: Invalid nonce in AJAX request', 'error');
            wp_send_json_error(['message' => 'Invalid nonce']);
        }
        
        // Check if order_id is provided
        if (!isset($_POST['order_id'])) {
            Loadlink_Logger::log('ERROR: No order_id provided in AJAX request', 'error');
            wp_send_json_error(['message' => 'No order ID provided']);
        }
        
        $order_id = intval($_POST['order_id']);
        if (!$order_id) {
            Loadlink_Logger::log('ERROR: Invalid order ID: ' . $_POST['order_id'], 'error');
            wp_send_json_error(['message' => 'Invalid order ID']);
        }
        
        Loadlink_Logger::log('AJAX order confirmation requested for order: ' . $order_id, 'info');
        
        // Process the order confirmation
        $result = self::process_order_confirmation($order_id);
        
        if ($result) {
            Loadlink_Logger::log('AJAX order confirmation successful for order: ' . $order_id, 'info');
            wp_send_json_success(['message' => 'Order confirmation processed successfully']);
        } else {
            Loadlink_Logger::log('AJAX order confirmation failed for order: ' . $order_id, 'error');
            wp_send_json_error(['message' => 'Order confirmation failed']);
        }
    }
    
    /**
     * Test function to manually save freight data to an order
     */
    public static function ajax_test_freight_save() {
        $order_id = intval($_POST['order_id']);
        if (!$order_id) {
            wp_send_json_error('Invalid order ID');
        }
        
        Loadlink_Logger::log('=== MANUAL TEST FREIGHT SAVE ===', 'info');
        Loadlink_Logger::log('Testing freight save for order: ' . $order_id, 'info');
        
        // Call the save function directly
        self::save_freight_data_to_order($order_id);
        
        // Check if freight data was saved
        $order = wc_get_order($order_id);
        $freight_data = $order->get_meta('_loadlink_freight_data');
        
        if (!empty($freight_data)) {
            wp_send_json_success([
                'message' => 'Freight data saved successfully',
                'freight_data' => $freight_data
            ]);
        } else {
            wp_send_json_error('Freight data was not saved');
        }
    }
}
