<?php
/**
 * Loadlink Order Actions
 */

if (!defined('ABSPATH')) {
    exit;
}

class Loadlink_Order_Actions {
    
    public function __construct() {
        add_action('woocommerce_order_actions', [$this, 'add_order_actions']);
        add_action('woocommerce_order_action_loadlink_view_freight', [$this, 'handle_view_freight_action']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
        add_action('wp_ajax_loadlink_get_order_freight_details', [$this, 'ajax_get_order_freight_details']);
    }
    
    /**
     * Add custom actions to WooCommerce orders
     */
    public function add_order_actions($actions) {
        global $post;
        
        if (!$post) {
            return $actions;
        }
        
        $order = wc_get_order($post->ID);
        if (!$order) {
            return $actions;
        }
        
        // Check if this order has Loadlink freight data
        $freight_data = $order->get_meta('_loadlink_freight_data');
        if (empty($freight_data)) {
            return $actions;
        }
        
        // Add View Freight Details action
        $actions['loadlink_view_freight'] = __('View Freight Details', 'loadlink');
        
        return $actions;
    }
    
    /**
     * Handle the view freight action
     */
    public function handle_view_freight_action($order) {
        if (!$order) {
            return;
        }
        
        $freight_data = $order->get_meta('_loadlink_freight_data');
        if (empty($freight_data)) {
            return;
        }
        
        // Redirect to the freight details page with order ID
        $freight_id = $freight_data['freight_id'] ?? '';
        $redirect_url = admin_url('admin.php?page=loadlink-freight&view_freight=' . $freight_id . '&order_id=' . $order->get_id());
        
        wp_redirect($redirect_url);
        exit;
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'post.php' || !isset($_GET['post'])) {
            return;
        }
        
        $post_id = intval($_GET['post']);
        $post = get_post($post_id);
        
        if (!$post || $post->post_type !== 'shop_order') {
            return;
        }
        
        $order = wc_get_order($post_id);
        if (!$order) {
            return;
        }
        
        $freight_data = $order->get_meta('_loadlink_freight_data');
        if (empty($freight_data)) {
            return;
        }
        
        wp_enqueue_style('loadlink-order-actions', LOADLINK_PLUGIN_URL . 'assets/order-actions.css', [], LOADLINK_VERSION);
        wp_enqueue_script('loadlink-order-actions', LOADLINK_PLUGIN_URL . 'assets/order-actions.js', ['jquery'], LOADLINK_VERSION, true);
        
        wp_localize_script('loadlink-order-actions', 'loadlinkOrderActions', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('loadlink_order_actions_nonce'),
            'orderId' => $order->get_id(),
            'freightId' => $freight_data['freight_id'] ?? ''
        ]);
    }
    
    /**
     * AJAX handler for order freight details
     */
    public function ajax_get_order_freight_details() {
        check_ajax_referer('loadlink_order_actions_nonce', 'nonce');
        
        $order_id = intval($_POST['order_id']);
        $freight_id = sanitize_text_field($_POST['freight_id']);
        
        if (!$order_id || !$freight_id) {
            wp_send_json_error('Invalid parameters');
        }
        
        $order = wc_get_order($order_id);
        if (!$order) {
            wp_send_json_error('Order not found');
        }
        
        $freight_data = $order->get_meta('_loadlink_freight_data');
        if (!$freight_data || $freight_data['freight_id'] != $freight_id) {
            wp_send_json_error('Freight data not found');
        }
        
        $details = $this->format_order_freight_details($freight_data, $order);
        
        wp_send_json_success($details);
    }
    
    /**
     * Format freight details for order page display
     */
    private function format_order_freight_details($freight_data, $order) {
        $html = '<div class="loadlink-order-freight-details">';
        
        // Basic info
        $html .= '<div class="loadlink-detail-section">';
        $html .= '<h3>Freight Information</h3>';
        $html .= '<table class="loadlink-details-table">';
        $html .= '<tr><td><strong>Freight ID:</strong></td><td>' . esc_html($freight_data['freight_id']) . '</td></tr>';
        $html .= '<tr><td><strong>Order Number:</strong></td><td>' . esc_html($order->get_order_number()) . '</td></tr>';
        $html .= '<tr><td><strong>Order Date:</strong></td><td>' . esc_html($order->get_date_created()->date('d/m/Y H:i')) . '</td></tr>';
        $html .= '<tr><td><strong>Status:</strong></td><td>' . esc_html(wc_get_order_status_name($order->get_status())) . '</td></tr>';
        $html .= '</table>';
        $html .= '</div>';
        
        // Pickup address
        if (!empty($freight_data['pickup_address'])) {
            $html .= '<div class="loadlink-detail-section">';
            $html .= '<h3>Pickup Address</h3>';
            $html .= '<div class="loadlink-address">';
            $html .= '<p><strong>' . esc_html($freight_data['pickup_address']['name'] ?? '') . '</strong></p>';
            $html .= '<p>' . esc_html($freight_data['pickup_address']['address'] ?? '') . '</p>';
            $html .= '<p>' . esc_html($freight_data['pickup_address']['suburb'] ?? '') . ' ' . esc_html($freight_data['pickup_address']['postcode'] ?? '') . '</p>';
            $html .= '<p>' . esc_html($freight_data['pickup_address']['state'] ?? '') . '</p>';
            $html .= '</div>';
            $html .= '</div>';
        }
        
        // Delivery address
        if (!empty($freight_data['delivery_address'])) {
            $html .= '<div class="loadlink-detail-section">';
            $html .= '<h3>Delivery Address</h3>';
            $html .= '<div class="loadlink-address">';
            $html .= '<p><strong>' . esc_html($freight_data['delivery_address']['name'] ?? '') . '</strong></p>';
            $html .= '<p>' . esc_html($freight_data['delivery_address']['address'] ?? '') . '</p>';
            $html .= '<p>' . esc_html($freight_data['delivery_address']['suburb'] ?? '') . ' ' . esc_html($freight_data['delivery_address']['postcode'] ?? '') . '</p>';
            $html .= '<p>' . esc_html($freight_data['delivery_address']['state'] ?? '') . '</p>';
            $html .= '</div>';
            $html .= '</div>';
        }
        
        // Service details
        if (!empty($freight_data['selected_service_quote_id'])) {
            $html .= '<div class="loadlink-detail-section">';
            $html .= '<h3>Service Details</h3>';
            $html .= '<table class="loadlink-details-table">';
            $html .= '<tr><td><strong>Service:</strong></td><td>' . esc_html($freight_data['selected_service_name'] ?? 'Unknown') . '</td></tr>';
            $html .= '<tr><td><strong>Quote ID:</strong></td><td>' . esc_html($freight_data['selected_service_quote_id']) . '</td></tr>';
            $html .= '</table>';
            $html .= '</div>';
        }
        
        // API Response details
        if (!empty($freight_data['api_response'])) {
            $html .= '<div class="loadlink-detail-section">';
            $html .= '<h3>API Response</h3>';
            $html .= '<div class="loadlink-api-response">';
            $html .= '<pre>' . esc_html(json_encode($freight_data['api_response'], JSON_PRETTY_PRINT)) . '</pre>';
            $html .= '</div>';
            $html .= '</div>';
        }
        
        $html .= '</div>';
        
        return $html;
    }
}
