<?php
class Loadlink_Client {
    private $api_url;
    private $api_key;
    private $api_secret;
    private $user_id;
    private $timeout = 30;
    private static $cache = [];
    private static $cache_timeout = 300; // 5 minutes
    private static $pending_requests = [];
    private static $freight_data = []; // Store freight_id and service_quote_ids
    private static $last_api_call_time = 0;
    private static $last_api_payload_hash = '';
    
    public function __construct() {
        $this->api_url = get_option('loadlink_api_url', 'https://parcelfreight.loadlink.net.au/api/v1/partner/freight/get_price');
        $this->api_key = get_option('loadlink_api_key', '');
        $this->api_secret = get_option('loadlink_api_secret', '');
        $this->user_id = get_option('loadlink_user_id', 56); // Default for testing
    }
    
	public function fetch_rates($payload) {
        try {
			// IMPORTANT: Build the final API payload first so our cache/throttle keys
			// are stable and based on actual request content (prevents duplicate API calls)
			$api_payload = $this->build_api_payload($payload);

			// Check if address is complete before proceeding
			if (!$this->is_address_complete($payload)) {
				Loadlink_Logger::log('Address incomplete, skipping API call', 'debug');
				return [];
			}

			// Create cache key based on the built API payload
			$cache_key = $this->generate_cache_key($api_payload);

			// Session-scoped cache keyed by address + cart hash (prevents duplicate freights)
			$session = function_exists('WC') && WC()->session ? WC()->session : null;
			$session_cache_ttl = 600; // 10 minutes
			$session_cache_bucket_key = 'loadlink_rate_cache';
			// Create a stable session cache key based on address and dataRows content
			$address_fingerprint = [
				'pickup_suburb' => $api_payload['pickup_suburb'] ?? '',
				'delivery_suburb' => $api_payload['delivery_suburb'] ?? '',
				'pickup_building_type' => $api_payload['pickup_building_type'] ?? '',
				'delivery_building_type' => $api_payload['delivery_building_type'] ?? ''
			];
			
			// Sort dataRows for consistent hashing
			$dataRows = $api_payload['dataRows'] ?? [];
			if (is_array($dataRows)) {
				usort($dataRows, function($a, $b) {
					return strcmp(serialize($a), serialize($b));
				});
			}
			
			$session_cache_key = md5(json_encode(['address' => $address_fingerprint, 'dataRows' => $dataRows]));

			if ($session) {
				$bucket = $session->get($session_cache_bucket_key);
				$bucket = is_array($bucket) ? $bucket : [];
				if (isset($bucket[$session_cache_key])) {
					$entry = $bucket[$session_cache_key];
					if (isset($entry['timestamp']) && (time() - (int) $entry['timestamp']) < $session_cache_ttl) {
						Loadlink_Logger::log('Session cache hit for key: ' . $session_cache_key, 'debug');
						// Restore freight data if present
						if (isset($entry['freight_data'])) {
							self::$freight_data = $entry['freight_data'];
						}
						return $entry['rates'];
					} else {
						Loadlink_Logger::log('Session cache stale or missing timestamp for key: ' . $session_cache_key, 'debug');
					}
				}
			}

			// Persistent cache (session/transient) first to avoid duplicate freights on refresh
			$persist = $this->get_persistent_cache($cache_key);
			if ($persist) {
				// Restore freight data if present
				if (isset($persist['freight_data'])) {
					self::$freight_data = $persist['freight_data'];
				}
				Loadlink_Logger::log('Returning rates from persistent cache for key: ' . $cache_key, 'debug');
				return $persist['rates'];
			}

			// Enhanced duplicate call prevention using transients (works across PHP processes)
			$payload_hash = md5(json_encode($api_payload));
			$now = time();
			$throttle_ttl = 10; // 10 seconds for better duplicate prevention
			$disable_throttling = get_option('loadlink_disable_throttling', 0);
			
			// Check transient-based duplicate prevention (works across processes)
			$duplicate_key = 'loadlink_duplicate_' . $payload_hash;
			$last_call_data = get_transient($duplicate_key);
			
			if (!$disable_throttling && $last_call_data && 
				isset($last_call_data['timestamp']) && 
				($now - $last_call_data['timestamp']) < $throttle_ttl) {
				Loadlink_Logger::log('Duplicate API call prevented via transient - same payload within throttle window (payload hash: ' . $payload_hash . ', time diff: ' . ($now - $last_call_data['timestamp']) . 's)', 'info');
				// Return cached rates if available
				if (isset(self::$cache[$cache_key])) {
					return self::$cache[$cache_key]['rates'];
				}
				// Try to get from persistent cache
				$persist = $this->get_persistent_cache($cache_key);
				if ($persist) {
					return $persist['rates'];
				}
				return [];
			}
			
			// Session-based throttling for additional protection
			$session = function_exists('WC') && WC()->session ? WC()->session : null;
			
			if ($session && !$disable_throttling) {
				$last_request_key = 'loadlink_last_request_' . $cache_key;
				$last_request_time = $session->get($last_request_key);
				if (is_numeric($last_request_time) && ($now - (int) $last_request_time) < $throttle_ttl) {
					// If we have a recent request for the same payload, prefer cached data or skip
					if (isset(self::$cache[$cache_key])) {
						$cached_data = self::$cache[$cache_key];
						if ($now - $cached_data['timestamp'] < self::$cache_timeout) {
							Loadlink_Logger::log('Throttled: returning cached rates for key: ' . $cache_key, 'debug');
							// Restore freight data if present
							if (isset($cached_data['freight_data'])) {
								self::$freight_data = $cached_data['freight_data'];
							}
							return $cached_data['rates'];
						}
					}
					Loadlink_Logger::log('Throttled: skipping API call for key: ' . $cache_key, 'debug');
					return isset(self::$cache[$cache_key]) ? self::$cache[$cache_key]['rates'] : [];
				}
				// Record request time early to collapse bursts
				$session->set($last_request_key, $now);
			}
            
            // Check cache first
            if (isset(self::$cache[$cache_key])) {
                $cached_data = self::$cache[$cache_key];
                if (time() - $cached_data['timestamp'] < self::$cache_timeout) {
                    Loadlink_Logger::log('Memory cache hit - returning cached rates for key: ' . $cache_key, 'info');
                    // Restore freight data from cache
                    if (isset($cached_data['freight_data'])) {
                        self::$freight_data = $cached_data['freight_data'];
                        Loadlink_Logger::log('Restored freight data from cache: ' . json_encode(self::$freight_data), 'debug');
                    }
                    return $cached_data['rates'];
                } else {
                    // Cache expired, remove it
                    unset(self::$cache[$cache_key]);
                    Loadlink_Logger::log('Memory cache expired for key: ' . $cache_key, 'debug');
                }
            }
            
            // Check if there's already a pending request for this cache key (local)
            if (isset(self::$pending_requests[$cache_key])) {
                Loadlink_Logger::log('Request already pending for key: ' . $cache_key . ', waiting...', 'debug');
                // Wait for the pending request to complete
                $start_time = time();
                while (isset(self::$pending_requests[$cache_key]) && (time() - $start_time) < 30) {
                    usleep(100000); // Wait 100ms
                }
                
                // Check cache again after waiting
                if (isset(self::$cache[$cache_key])) {
                    $cached_data = self::$cache[$cache_key];
                    if (time() - $cached_data['timestamp'] < self::$cache_timeout) {
                        Loadlink_Logger::log('Returning cached rates after waiting for key: ' . $cache_key, 'debug');
                        // Restore freight data from cache
                        if (isset($cached_data['freight_data'])) {
                            self::$freight_data = $cached_data['freight_data'];
                            Loadlink_Logger::log('Restored freight data after waiting: ' . json_encode(self::$freight_data), 'debug');
                        }
                        return $cached_data['rates'];
                    }
                }
            }
            
            // Check if there's a pending request for the same payload across processes
            $pending_key = 'loadlink_pending_' . $payload_hash;
            $pending_data = get_transient($pending_key);
            if ($pending_data && isset($pending_data['timestamp']) && (time() - $pending_data['timestamp']) < 30) {
                Loadlink_Logger::log('Request already pending across processes for payload hash: ' . $payload_hash . ', waiting...', 'debug');
                // Wait for the pending request to complete
                $start_time = time();
                while (get_transient($pending_key) && (time() - $start_time) < 30) {
                    usleep(100000); // Wait 100ms
                }
                
                // Check if results are now available
                $persist = $this->get_persistent_cache($cache_key);
                if ($persist) {
                    Loadlink_Logger::log('Returning rates from persistent cache after waiting for pending request', 'debug');
                    return $persist['rates'];
                }
            }
            
			// Mark this request as pending (both local and transient)
            self::$pending_requests[$cache_key] = true;
            set_transient($pending_key, ['timestamp' => time(), 'cache_key' => $cache_key], 30);
            
            // If mock mode is enabled, return simulated response
			if ((int) get_option('loadlink_mock_mode', 0) === 1) {
                $mock = $this->get_mock_response();
                $rates = $this->parse_api_response(json_encode($mock));
                // Cache the mock response
                self::$cache[$cache_key] = [
                    'rates' => $rates,
                    'timestamp' => time()
                ];
				// Store in session cache as well
				if ($session) {
					$bucket = $session->get($session_cache_bucket_key);
					$bucket = is_array($bucket) ? $bucket : [];
					$bucket[$session_cache_key] = [
						'rates' => $rates,
						'freight_data' => self::$freight_data,
						'timestamp' => time()
					];
					// Prune if oversized
					if (count($bucket) > 20) {
						array_shift($bucket);
					}
					$session->set($session_cache_bucket_key, $bucket);
				}
                return $rates;
            }
            
            // Validate API credentials
            if (empty($this->api_key) || empty($this->api_secret)) {
                $error_msg = 'Loadlink API credentials not configured. Please check Settings → Loadlink.';
                Loadlink_Logger::log($error_msg, 'error');
                $this->add_admin_notice($error_msg, 'error');
                return [];
            }
            
			// Record this API call to prevent duplicates (both static and transient)
			self::$last_api_call_time = $now;
			self::$last_api_payload_hash = $payload_hash;
			
			// Set transient to prevent duplicates across processes
			set_transient($duplicate_key, ['timestamp' => $now, 'payload_hash' => $payload_hash], $throttle_ttl);
			
			Loadlink_Logger::log('Making API call - payload hash: ' . $payload_hash . ', cache key: ' . $cache_key, 'info');
			$response = $this->make_api_request($api_payload, $cache_key);
            
            if (is_wp_error($response)) {
                $error_msg = 'Loadlink API request failed: ' . $response->get_error_message();
                Loadlink_Logger::log($error_msg, 'error');
                $this->add_admin_notice($error_msg, 'error');
                return [];
            }
            
            $rates = $this->parse_api_response($response);
            
            // If no rates returned, show error instead of fallback
            if (empty($rates)) {
                $error_msg = 'No shipping rates returned from Loadlink API. Please check your configuration.';
                Loadlink_Logger::log($error_msg, 'warning');
                $this->add_admin_notice($error_msg, 'warning');
                return [];
            }
            
			// Cache the successful response
			self::$cache[$cache_key] = [
                'rates' => $rates,
                'freight_data' => self::$freight_data,
                'timestamp' => time()
            ];
			// Also store persistent cache to survive page refresh and PHP process changes
			$this->set_persistent_cache($cache_key, [
				'rates' => $rates,
				'freight_data' => self::$freight_data,
				'timestamp' => time()
			]);

			// Store in session-scoped cache bucket keyed by address+cart
			if ($session) {
				$bucket = $session->get($session_cache_bucket_key);
				$bucket = is_array($bucket) ? $bucket : [];
				$bucket[$session_cache_key] = [
					'rates' => $rates,
					'freight_data' => self::$freight_data,
					'timestamp' => time()
				];
				// Prune oldest if bucket grows too large
				if (count($bucket) > 20) {
					array_shift($bucket);
				}
				$session->set($session_cache_bucket_key, $bucket);
			}
            
            Loadlink_Logger::log('Cached rates for key: ' . $cache_key, 'debug');
            return $rates;
            
		} catch (Exception $e) {
            $error_msg = 'Loadlink API exception: ' . $e->getMessage();
            Loadlink_Logger::log($error_msg, 'error');
            $this->add_admin_notice($error_msg, 'error');
            return [];
        } finally {
            // Always clean up pending request (both local and transient)
            if (isset($cache_key)) {
                unset(self::$pending_requests[$cache_key]);
            }
            if (isset($pending_key)) {
                delete_transient($pending_key);
            }
        }
    }

    private function get_persistent_cache($cache_key) {
		$now = time();
		$ttl = self::$cache_timeout;
		// Check session
		if (function_exists('WC') && WC()->session) {
			$session_key = 'loadlink_cache_' . $cache_key;
			$entry = WC()->session->get($session_key);
			if (is_array($entry) && isset($entry['timestamp']) && ($now - (int) $entry['timestamp'] < $ttl)) {
				return $entry;
			}
		}
		// Check site transient (shared across requests)
		$transient_key = 'loadlink_' . $cache_key;
		$entry = get_transient($transient_key);
		if (is_array($entry) && isset($entry['timestamp']) && ($now - (int) $entry['timestamp'] < $ttl)) {
			return $entry;
		}
		return null;
	}

	private function set_persistent_cache($cache_key, $data) {
		$ttl = self::$cache_timeout;
		if (function_exists('WC') && WC()->session) {
			$session_key = 'loadlink_cache_' . $cache_key;
			WC()->session->set($session_key, $data);
		}
		$transient_key = 'loadlink_' . $cache_key;
		set_transient($transient_key, $data, $ttl);
    }

    private function get_mock_response() {
        return [
            'road express' => [
                'type' => 'Couriers Please',
                'name' => 'road express',
                'code' => 'L55',
                'quote' => '38.57',
                'user_price' => '44.36',
                'gst_price' => 4.44,
                'currency' => 'AUD',
                'collection_cutoff_time' => '15:59',
                'estimated_delivery_datetime' => '2025-10-16 04:49:13',
                'service_quote_id' => 734
            ],
            'overnight express' => [
                'type' => 'Fed Ex',
                'name' => 'overnight express',
                'code' => '75',
                'quote' => '134.2',
                'user_price' => '154.33',
                'gst_price' => 15.43,
                'currency' => 'AUD',
                'collection_cutoff_time' => '15:00:00',
                'estimated_delivery_datetime' => '2025-10-10T17:00:00',
                'service_quote_id' => 735
            ],
            'air express' => [],
            'asap hot shot' => [],
            'technology express' => [
                'type' => 'Fed Ex',
                'name' => 'technology express',
                'code' => '717B',
                'quote' => '84.48',
                'user_price' => '97.15',
                'gst_price' => 9.72,
                'currency' => 'AUD',
                'collection_cutoff_time' => '15:00:00',
                'estimated_delivery_datetime' => '2025-10-14T17:00:00',
                'service_quote_id' => 736
            ],
            'freight_id' => 644,
            'freight_created_at' => '2025-10-07T04:49:11.000000Z'
        ];
    }
    
    public function build_api_payload($woocommerce_package) {
        $destination = $woocommerce_package['destination'];
        $contents = $woocommerce_package['contents'];
        
        // Build address strings in Loadlink format CITY, POSTCODE, STATE
        $base_location = function_exists('wc_get_base_location') ? wc_get_base_location() : ['country' => get_option('woocommerce_default_country'), 'state' => ''];
        $pickup_city = get_option('woocommerce_store_city');
        $pickup_postcode = get_option('woocommerce_store_postcode');
        $pickup_country = isset($base_location['country']) ? $base_location['country'] : '';
        $pickup_state = isset($base_location['state']) ? $base_location['state'] : '';

        $pickup_suburb = $this->format_address($pickup_city, $pickup_postcode, $pickup_country, $pickup_state);
        $delivery_suburb = $this->format_address($destination['city'] ?? '', $destination['postcode'] ?? '', $destination['country'] ?? '', $destination['state'] ?? '');
        
        // Add phone fields for pickup (store)
        $pickup_phone_raw = get_option('woocommerce_store_phone', '');
        $pickup_parsed_phone = class_exists('Loadlink_Order_Confirmation')
            ? Loadlink_Order_Confirmation::parse_phone_number($pickup_phone_raw, 'AU')
            : ['area_code' => '', 'phone' => $pickup_phone_raw];
        $pickup_phone = $pickup_parsed_phone['phone'];
        $pickup_phone_area_code = $pickup_parsed_phone['area_code'];

        // Add phone fields for dropoff (customer)
        $dropoff_phone_raw = '';
        if (!empty($destination['shipping_phone'])) {
            $dropoff_phone_raw = $destination['shipping_phone'];
        } elseif (!empty($destination['billing_phone'])) {
            $dropoff_phone_raw = $destination['billing_phone'];
        } elseif (!empty($destination['phone'])) {
            $dropoff_phone_raw = $destination['phone'];
        }
        
        // Fallback: try session (Blocks or classic) or _loadlink_dropoff_phone order meta
        if (empty($dropoff_phone_raw) && function_exists('WC') && WC()->session) {
            $dropoff_phone_from_session = WC()->session->get('loadlink_dropoff_phone');
            if (!empty($dropoff_phone_from_session)) {
                $dropoff_phone_raw = $dropoff_phone_from_session;
            }
        }
        // Fallback: if order_id known in $woocommerce_package
        if (empty($dropoff_phone_raw) && !empty($woocommerce_package['order_id'])) {
            $order = wc_get_order($woocommerce_package['order_id']);
            if ($order) {
                $meta_phone = $order->get_meta('_loadlink_dropoff_phone');
                if (!empty($meta_phone)) {
                    $dropoff_phone_raw = $meta_phone;
                }
            }
        }
        $dropoff_parsed_phone = class_exists('Loadlink_Order_Confirmation')
            ? Loadlink_Order_Confirmation::parse_phone_number($dropoff_phone_raw, 'AU')
            : ['area_code' => '', 'phone' => $dropoff_phone_raw];
        $dropoff_phone = $dropoff_parsed_phone['phone'];
        $dropoff_phone_area_code = $dropoff_parsed_phone['area_code'];

        // Store phone required check
        if (empty($pickup_phone)) {
            $notice = 'Store phone number is missing. Please contact the store administrator.';
            self::add_checkout_notice($notice);
            Loadlink_Logger::log('No store phone in WooCommerce settings. Cannot build payload.', 'warning');
            return [];
        }

        // Check for products with missing dimensions
        $missing_dimensions = [];
        foreach ($contents as $item) {
            $product = $item['data'];
            $product_name = $product->get_name();
            
            // Check if any dimensions are missing or zero
            if (empty($product->get_length()) || empty($product->get_width()) || empty($product->get_height()) || empty($product->get_weight())) {
                $missing_dimensions[] = $product_name;
            }
        }
        
        // If products have missing dimensions, add admin notice and return empty rates
        if (!empty($missing_dimensions)) {
            $missing_products = implode(', ', $missing_dimensions);
            $error_msg = "Loadlink: The following products are missing dimensions or weight: {$missing_products}. Please add dimensions to these products to get shipping rates.";
            $checkout_msg = "Some products in your cart are missing dimensions. Please contact us for shipping rates.";
            
            Loadlink_Logger::log($error_msg, 'warning');
            $this->add_admin_notice($error_msg, 'warning');
            self::add_checkout_notice($checkout_msg);
            return [];
        }
        
        // Build dataRows array - create one row per product with total quantity
        $dataRows = [];
		foreach ($contents as $item) {
            $product = $item['data'];
            $quantity = $item['quantity'];
            
            // Get product dimensions and weight - ensure they are numeric
            $raw_length = $product->get_length();
            $raw_width = $product->get_width();
            $raw_height = $product->get_height();
            $raw_weight = $product->get_weight();
            
            Loadlink_Logger::log("Product: {$product->get_name()}", 'debug');
            Loadlink_Logger::log("Raw dimensions - Length: " . var_export($raw_length, true) . " (type: " . gettype($raw_length) . ")", 'debug');
            Loadlink_Logger::log("Raw dimensions - Width: " . var_export($raw_width, true) . " (type: " . gettype($raw_width) . ")", 'debug');
            Loadlink_Logger::log("Raw dimensions - Height: " . var_export($raw_height, true) . " (type: " . gettype($raw_height) . ")", 'debug');
            Loadlink_Logger::log("Raw dimensions - Weight: " . var_export($raw_weight, true) . " (type: " . gettype($raw_weight) . ")", 'debug');
            
            $length = floatval($raw_length ?: 3);
            $width = floatval($raw_width ?: 3); 
            $height = floatval($raw_height ?: 3);
            $weight = floatval($raw_weight ?: 9.00);
            
            Loadlink_Logger::log("Converted dimensions - Length: {$length}, Width: {$width}, Height: {$height}, Weight: {$weight}", 'debug');
            
            // Validate dimensions are positive numbers
            if ($length <= 0 || $width <= 0 || $height <= 0 || $weight <= 0) {
                Loadlink_Logger::log('Invalid product dimensions: length=' . $length . ', width=' . $width . ', height=' . $height . ', weight=' . $weight, 'warning');
                // Skip this product if dimensions are invalid
                continue;
            }
            
			// Packaging code per product (fallback to global setting if empty)
			$packaging_code = '';
			if (is_callable([$product, 'get_meta'])) {
				$packaging_code = (string) $product->get_meta('_loadlink_packaging_code');
			}

			// Create one dataRow per product - use new object format
            $quantity_int = (int) max(1, (int) $quantity);
            $weight_float = (float) $weight;
            $length_int = (int) round($length);
            $width_int = (int) round($width);
            $height_int = (int) round($height);
            $volume = $length_int * $width_int * $height_int;
            
            $dataRow = [
                'packagingType' => (string) ($packaging_code ?: get_option('loadlink_packaging_code', 'BX')),
                'qty'           => $quantity_int,
                'kgs'           => round($weight_float, 2), // two-decimal precision
                'length'        => $length_int,
                'width'         => $width_int,
                'height'        => $height_int,
                'weight'        => round($weight_float * $quantity_int, 2), // preserve decimal precision
                'volume'        => $volume * $quantity_int
            ];

            
            // Validate dataRow has all required fields
            $required_fields = ['packagingType', 'qty', 'kgs', 'length', 'width', 'height', 'weight', 'volume'];
            $missing_fields = [];
            foreach ($required_fields as $field) {
                if (!isset($dataRow[$field]) || $dataRow[$field] === null) {
                    $missing_fields[] = $field;
                } elseif ($field === 'packagingType') {
                    // packagingType should be a non-empty string
                    if (!is_string($dataRow[$field]) || empty($dataRow[$field])) {
                        $missing_fields[] = $field . ' (must be non-empty string)';
                    }
                } else {
                    // Other fields should be numeric
                    if (!is_numeric($dataRow[$field])) {
                        $missing_fields[] = $field . ' (must be numeric)';
                    }
                }
            }
            
            if (!empty($missing_fields)) {
                Loadlink_Logger::log("ERROR: Missing required fields in dataRow for product '{$product->get_name()}': " . implode(', ', $missing_fields), 'error');
                continue; // Skip this product
            }
            
            // Log the dataRow for debugging
            Loadlink_Logger::log("Creating dataRow for product '{$product->get_name()}' with quantity {$quantity}: " . json_encode($dataRow), 'debug');
            
			$dataRows[] = $dataRow;
        }
        
		// Prefer user-selected dropoff building type from session, fallback to settings
		$session_dropoff = function_exists('WC') && WC()->session ? WC()->session->get('loadlink_dropoff_building_type') : '';
		$delivery_building_type = in_array($session_dropoff, ['Residential', 'Commercial'], true)
			? $session_dropoff
			: get_option('loadlink_delivery_building_type', 'Residential');

		// Ensure dataRows is always an array, never null
		if (!is_array($dataRows)) {
			Loadlink_Logger::log('WARNING: dataRows is not an array, converting to empty array', 'warning');
			$dataRows = [];
		}
		
		// Validate that we have at least one dataRow
		if (empty($dataRows)) {
			Loadlink_Logger::log('ERROR: No dataRows to send to API - this will cause API errors', 'error');
			return [];
		}

		$payload = [
            'pickup_suburb' => (string) $pickup_suburb,
            'delivery_suburb' => (string) $delivery_suburb,
			'delivery_building_type' => (string) $delivery_building_type,
            'pickup_building_type' => (string) get_option('loadlink_pickup_building_type', 'Residential'),
            'pickup_phone' => $pickup_phone,
            'pickup_phone_area_code' => $pickup_phone_area_code,
            'dropoff_phone' => $dropoff_phone,
            'dropoff_phone_area_code' => $dropoff_phone_area_code,
            'dataRows' => $dataRows,
            'is_danger' => (bool) false,
            'user_id' => (int) $this->user_id
        ];
        
        Loadlink_Logger::log('API payload: ' . json_encode($payload, JSON_PRETTY_PRINT), 'info');
        
        return $payload;
    }
    
    private function format_address($city, $postcode, $country, $state) {
        // For Loadlink, the expected format appears to be CITY, POSTCODE, STATE (AU states like QLD, NSW)
        $parts = [
            strtoupper(trim((string)$city)),
            trim((string)$postcode),
            $this->normalize_state($state)
        ];
        // Remove empty parts and join with comma+space
        $parts = array_values(array_filter($parts, function($v) { return $v !== ''; }));
        return implode(', ', $parts);
    }

    private function normalize_state($state) {
        $state = strtoupper(trim((string)$state));
        $map = [
            'VICTORIA' => 'VIC',
            'NEW SOUTH WALES' => 'NSW',
            'QUEENSLAND' => 'QLD',
            'SOUTH AUSTRALIA' => 'SA',
            'WESTERN AUSTRALIA' => 'WA',
            'TASMANIA' => 'TAS',
            'NORTHERN TERRITORY' => 'NT',
            'AUSTRALIAN CAPITAL TERRITORY' => 'ACT'
        ];
        return isset($map[$state]) ? $map[$state] : $state;
    }
    
    public function make_api_request($payload, $idempotency_key = null) {
        // Generate timestamp for request signing
        $timestamp = time();
        
        // Create signature (common pattern for API authentication)
        $signature_string = $this->api_key . $timestamp . json_encode($payload);
        $signature = hash_hmac('sha256', $signature_string, $this->api_secret);
        
        $args = [
            'method' => 'POST',
            'timeout' => $this->timeout,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'X-API-Key' => $this->api_key,
                'X-API-Secret' => $this->api_secret,
                'X-Timestamp' => $timestamp,
                'X-Signature' => $signature
            ],
            'body' => json_encode($payload)
        ];

        // Add idempotency header to let API safely ignore duplicates
        if (!empty($idempotency_key)) {
            $args['headers']['Idempotency-Key'] = $idempotency_key;
        }
        
        Loadlink_Logger::log('API URL: ' . $this->api_url, 'info');
        
        
        
        $response = wp_remote_post($this->api_url, $args);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        Loadlink_Logger::log('API response: ' . $response_body, 'info');
        
        if ($response_code !== 200) {
            return new WP_Error('api_error', 'API returned status code: ' . $response_code . ' - ' . $response_body);
        }
        
        return $response_body;
    }
    
    public function confirm_freight_order($service_quote_id, $order_data, $freight_id) {
        // Add a small delay to ensure the freight is fully processed on the API side
        Loadlink_Logger::log('Adding 2-second delay before confirmation to ensure freight is processed', 'debug');
        sleep(2);
        
        // Build the confirmation payload
        $payload = $this->build_confirmation_payload($service_quote_id, $order_data);
        
        // Make the API request
        $response = $this->make_confirmation_request($payload, $freight_id);
        
        return $response;
    }
    
    private function build_confirmation_payload($service_quote_id, $order_data) {
        // Extract order information
        $order = $order_data['order'];
        $pickup_address = $order_data['pickup_address'];
        $dropoff_address = $order_data['dropoff_address'];
        
        // Use WooCommerce order post ID for both external identifiers (no random/time suffix)
        $order_id = $order->get_id();
        $external_order_id = (string) $order_id;
        $external_order_no = (string) $order_id;
        
        $payload = [
            'freight_service_quote_id' => (int) $service_quote_id,
            'external_order_id' => (string) $external_order_id,
            'external_order_no' => (string) $external_order_no,
            
            // Pickup address details
            'pickup_state' => (string) $pickup_address['state'],
            'pickup_suburb' => (string) $pickup_address['suburb'],
            'pickup_postcode' => (string) $pickup_address['postcode'],
            'pickup_address1' => (string) $pickup_address['address1'],
            'pickup_address2' => (string) $pickup_address['address2'],
            'pickup_address3' => (string) $pickup_address['address3'],
            'pickup_company_name' => (string) $pickup_address['company_name'],
            'pickup_company_contact_name' => (string) $pickup_address['contact_name'],
            'pickup_company_email' => (string) $pickup_address['email'],
            'pickup_company_phone_area_code' => (string) $pickup_address['phone_area_code'],
            'pickup_company_phone' => (string) $pickup_address['phone'],
            
            // Dropoff address details
            'dropoff_state' => (string) $dropoff_address['state'],
            'dropoff_suburb' => (string) $dropoff_address['suburb'],
            'dropoff_postcode' => (string) $dropoff_address['postcode'],
            'dropoff_address1' => (string) $dropoff_address['address1'],
            'dropoff_address2' => $dropoff_address['address2'] ? (string) $dropoff_address['address2'] : null,
            'dropoff_address3' => (string) $dropoff_address['address3'],
            'dropoff_company_name' => (string) $dropoff_address['company_name'],
            'dropoff_company_contact_name' => (string) $dropoff_address['contact_name'],
            'dropoff_company_email' => (string) $dropoff_address['email'],
            'dropoff_company_phone_area_code' => (string) $dropoff_address['phone_area_code'],
            'dropoff_company_phone' => (string) $dropoff_address['phone']
        ];
        
        Loadlink_Logger::log('Confirmation payload: ' . json_encode($payload, JSON_PRETTY_PRINT), 'info');
        
        return $payload;
    }
    
    private function make_confirmation_request($payload, $freight_id) {
        // Generate timestamp for request signing
        $timestamp = time();
        
        // Create signature
        $signature_string = $this->api_key . $timestamp . json_encode($payload);
        $signature = hash_hmac('sha256', $signature_string, $this->api_secret);
        
        // Build the confirmation endpoint URL with freight_id in path
        $confirmation_url = str_replace(['/freight/get_price'], '', $this->api_url) . '/' . $freight_id . '/service_quote_confirm_with_address';
        
        // Log the confirmation URL
        Loadlink_Logger::log('Confirmation URL: ' . $confirmation_url, 'info');
        
        $args = [
            'method' => 'POST',
            'timeout' => $this->timeout,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'X-API-Key' => $this->api_key,
                'X-API-Secret' => $this->api_secret,
                'X-Timestamp' => $timestamp,
                'X-Signature' => $signature
            ],
            'body' => json_encode($payload)
        ];
        
        
        $response = wp_remote_post($confirmation_url, $args);
        
        if (is_wp_error($response)) {
            Loadlink_Logger::log('Freight confirmation request failed: ' . $response->get_error_message(), 'error');
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        Loadlink_Logger::log('Confirmation response: ' . $response_body, 'info');
        
        if ($response_code !== 200) {
            Loadlink_Logger::log('Freight confirmation failed with status: ' . $response_code, 'error');
            return new WP_Error('confirmation_error', 'Freight confirmation failed with status: ' . $response_code . ' - ' . $response_body);
        }
        
        return $response_body;
    }

    public function parse_api_response($response_body) {
        $data = json_decode($response_body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            Loadlink_Logger::log('Failed to parse API response JSON', 'error');
            return [];
        }
        
        // Extract and store freight_id and service_quote_ids
        $freight_id = $data['freight_id'] ?? null;
        $freight_created_at = $data['freight_created_at'] ?? null;
        
        if ($freight_id) {
            Loadlink_Logger::log('Storing freight_id: ' . $freight_id, 'debug');
            self::$freight_data['freight_id'] = $freight_id;
            self::$freight_data['freight_created_at'] = $freight_created_at;
        }
        
        $rates = [];
        $service_mapping = [
            'road express' => 'Road Express',
            'overnight express' => 'Overnight Express',
            'air express' => 'Air Express',
            'asap hot shot' => 'ASAP Hot Shot',
            'technology express' => 'Technology Express'
        ];
        
        foreach ($service_mapping as $api_key => $display_name) {
            if (isset($data[$api_key]) && !empty($data[$api_key])) {
                $service = $data[$api_key];
                $service_quote_id = $service['service_quote_id'] ?? null;
                
                // Store service_quote_id for this service
                if ($service_quote_id) {
                    self::$freight_data['service_quote_ids'][$api_key] = $service_quote_id;
                    Loadlink_Logger::log('Storing service_quote_id for ' . $api_key . ': ' . $service_quote_id, 'debug');
                }
                
                $rates[] = [
                    'id' => 'loadlink_' . sanitize_title($api_key),
                    'label' => $display_name . ' (' . $service['type'] . ')',
                    'cost' => $service['user_price'],
                    'meta_data' => [
                        'service_code' => $service['code'],
                        'estimated_delivery' => $service['estimated_delivery_datetime'],
                        'cutoff_time' => $service['collection_cutoff_time'],
                        'quote_id' => $service_quote_id,
                        'freight_id' => $freight_id
                    ]
                ];
            }
        }
        
        // If no rates returned, return empty array
        if (empty($rates)) {
            Loadlink_Logger::log('No rates returned from API', 'warning');
            return [];
        }
        
        Loadlink_Logger::log('Successfully parsed ' . count($rates) . ' rates from API', 'info');
        Loadlink_Logger::log('Freight data stored: ' . json_encode(self::$freight_data), 'debug');
        return $rates;
    }
    
    private function add_admin_notice($message, $type = 'error') {
        // Add admin notice for API errors
        add_action('admin_notices', function() use ($message, $type) {
            $class = $type === 'error' ? 'notice notice-error' : 'notice notice-warning';
            echo '<div class="' . $class . '"><p><strong>Loadlink:</strong> ' . esc_html($message) . '</p></div>';
        });
    }
    
    public static function add_checkout_notice($message) {
        // Add checkout notice for missing dimensions
        wc_add_notice($message, 'notice');
    }
    
    private function generate_cache_key($payload) {
        // Create a stable cache key based on the payload content only
        // Don't include cart_hash as it changes between requests
        $key_data = [
            'pickup_suburb' => $payload['pickup_suburb'] ?? '',
            'delivery_suburb' => $payload['delivery_suburb'] ?? '',
            'pickup_building_type' => $payload['pickup_building_type'] ?? '',
            'delivery_building_type' => $payload['delivery_building_type'] ?? '',
            'dataRows' => $payload['dataRows'] ?? [],
            'user_id' => $this->user_id
        ];
        
        // Sort dataRows to ensure consistent ordering
        if (isset($key_data['dataRows']) && is_array($key_data['dataRows'])) {
            usort($key_data['dataRows'], function($a, $b) {
                return strcmp(serialize($a), serialize($b));
            });
        }
        
        return 'loadlink_' . md5(serialize($key_data));
    }
    
    public static function clear_cache() {
        // Clear all cached rates
        self::$cache = [];
        
        // Clear all Loadlink transients
        global $wpdb;
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_loadlink_%' OR option_name LIKE '_transient_timeout_loadlink_%'");
        
        Loadlink_Logger::log('Loadlink cache and transients cleared', 'info');
    }
    
    public static function get_freight_id() {
        return self::$freight_data['freight_id'] ?? null;
    }
    
    public static function get_service_quote_id($service_key) {
        return self::$freight_data['service_quote_ids'][$service_key] ?? null;
    }
    
    public static function get_all_freight_data() {
        return self::$freight_data;
    }
    
    public static function clear_freight_data() {
        self::$freight_data = [];
        Loadlink_Logger::log('Freight data cleared', 'info');
    }
    
    /**
     * Check if the address is complete enough to make an API call
     * Only make API calls when we have all required address fields
     */
    private function is_address_complete($payload) {
        $destination = $payload['destination'] ?? [];
        
        // Check for required address fields
        $required_fields = ['postcode', 'state', 'country', 'city'];
        foreach ($required_fields as $field) {
            if (empty($destination[$field])) {
                Loadlink_Logger::log("Address incomplete - missing {$field}", 'debug');
                return false;
            }
        }
        
        // Additional validation for postcode format (Australian postcodes are 4 digits)
        $postcode = $destination['postcode'];
        if (!preg_match('/^\d{4}$/', $postcode)) {
            Loadlink_Logger::log("Address incomplete - invalid postcode format: {$postcode}", 'debug');
            return false;
        }
        
        // Additional validation for Australian state codes
        $state = $this->normalize_state($destination['state']);
        $valid_states = ['NSW', 'VIC', 'QLD', 'SA', 'WA', 'TAS', 'NT', 'ACT'];
        if (!in_array($state, $valid_states)) {
            Loadlink_Logger::log("Address incomplete - invalid state code: {$state} (original: {$destination['state']})", 'debug');
            return false;
        }
        
        // Check if we have cart contents
        $contents = $payload['contents'] ?? [];
        if (empty($contents)) {
            Loadlink_Logger::log("Address incomplete - no cart contents", 'debug');
            return false;
        }
        
        Loadlink_Logger::log('Address is complete, proceeding with API call', 'debug');
        return true;
    }
}
