<?php
/**
 * Loadlink Admin Freight Orders Management
 */

if (!defined('ABSPATH')) {
    exit;
}

class Loadlink_Admin_Freight {
    
    public function __construct() {
        add_action('wp_ajax_loadlink_get_freight_details', [$this, 'ajax_get_freight_details']);
        add_action('wp_ajax_loadlink_create_test_freight', [$this, 'ajax_create_test_freight']);
        add_action('wp_ajax_loadlink_create_freight_for_order', [$this, 'ajax_create_freight_for_order']);
    }
    
    /**
     * Display the freight orders page
     */
    public function display_freight_page() {
        $freight_orders = $this->get_freight_orders();
        $pagination = $this->get_pagination_data($freight_orders);
        
        ?>
        <div class="wrap">
            <h1 class="screen-reader-text">My Parcel Freight</h1>
            <hr class="wp-header-end">
            
            <div class="loadlink-header">
                <div class="loadlink-logo">
                    <svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px"
                         viewBox="0 0 689 101.9" style="enable-background:new 0 0 689 101.9;" xml:space="preserve" width="200" height="30">
                    <style type="text/css">
                        .st0{fill:#1E3D5B;}
                        .st1{fill:#F37D48;}
                    </style>
                    <g>
                        <g>
                            <path class="st0" d="M193.6,77.5c-1.5-1-1.6-2.5-0.2-4.5l18.8-28.4c1.4-2,3.4-3.5,6.2-4.5c2.7-0.9,6.4-1.4,10.9-1.4l40,0.1
                                c4.6,0,7.6,0.5,9,1.4c1.4,1,1.4,2.5,0,4.5l-18.8,28.4c-1.4,2-3.4,3.5-6,4.5c-2.6,0.9-6.3,1.4-10.8,1.4l-40-0.1
                                C198.1,79,195.1,78.5,193.6,77.5z M223.1,69.2c0.6,0.4,1.8,0.6,3.7,0.6l3.5,0c2,0,3.5-0.2,4.6-0.6c1.1-0.4,2-1.2,2.8-2.4
                                l10.5-15.9c0.8-1.2,0.9-2,0.4-2.4c-0.5-0.4-1.8-0.6-3.7-0.6h-3.5c-1.9,0-3.4,0.2-4.6,0.6c-1.1,0.4-2.1,1.2-2.9,2.4l-10.5,15.9
                                C222.6,68,222.5,68.8,223.1,69.2z"/>
                            <polygon class="st1" points="458,69.9 478.6,38.8 452.7,38.8 426,79 470.1,79 481.1,69.9 		"/>
                            <path class="st0" d="M434.4,40.2c-1.4-1-4.3-1.5-9-1.5l-58.9,0l-11,9.2l11,0L345.9,79l52.9,0.1c4.5,0,8.1-0.5,10.8-1.4
                                c2.7-0.9,4.7-2.4,6.1-4.5l18.8-28.4C435.9,42.7,435.9,41.2,434.4,40.2z M404.5,51L394,66.9c-0.8,1.2-1.7,2-2.8,2.4
                                c-1.1,0.4-2.6,0.6-4.6,0.6l-8.6,0l14.6-22h8.6c2,0,3.2,0.3,3.8,0.7C405.4,49,405.3,49.8,404.5,51z"/>
                            <path class="st1" d="M563.6,79L552,64l-9.9,15l-25.9,0l26.6-40.2h18.7l11.6,15.1l10-15.1H609L582.4,79H563.6z"/>
                            <polygon class="st1" points="534.3,38.8 508.4,38.8 481.7,79 507.6,79 		"/>
                            <polygon class="st1" points="651.3,58.9 683.7,38.8 657.8,38.8 633.2,54.3 643.4,38.8 617.5,38.8 590.8,79 616.7,79 627,63.6 
                                631.1,79 656.9,79 		"/>
                            <polygon class="st0" points="343.8,38.8 307.8,38.8 261.6,79 300.4,79 311.4,69.9 297.3,69.9 317.9,51.8 314.6,79 339.9,79 		"/>
                            <polygon class="st0" points="165.7,69.9 186.2,38.8 160.3,38.8 133.7,79 177.8,79 188.8,69.9 		"/>
                        </g>
                        <g>
                            <g>
                                <path class="st1" d="M119.2,22.9c-6.6,1.8-13.5,4.1-20.7,6.9l6.9,1.4c5.5-2.6,11.1-5,16.7-7.2L119.2,22.9z"/>
                                <path class="st1" d="M45.4,58.2H62c6.5-5.2,13.5-10.1,20.9-14.7l-10.2-1.9C63.5,46.5,54.4,52,45.4,58.2z"/>
                                <path class="st1" d="M134.9,19.5c-3.4,0.5-6.9,1.2-10.6,2.1l2.1,0.8C129.3,21.3,132.1,20.4,134.9,19.5z"/>
                                <path class="st1" d="M77.8,39l9.7,1.7c4.4-2.6,8.9-5,13.5-7.3l-7.5-1.6C88.3,33.9,83.1,36.3,77.8,39z"/>
                            </g>
                            <g>
                                <path class="st1" d="M38.1,63.4c-0.5,0.4-1,0.7-1.5,1.1C25.3,73.2,14.3,82.9,4.4,93.7h24c4.9-7.3,17.3-20.9,27.3-30.3H38.1z"/>
                            </g>
                            <path class="st0" d="M64.2,93.7c14.6-24.7,35.9-47.8,69.9-67.8c-45.7,12.5-84,55.4-91.5,67.8H64.2z"/>
                            <g>
                                <path class="st0" d="M120.3,34.4c-3,2-6,4-8.8,6.1c1,3.2,1.5,6.5,1.5,10c0,18.9-15.3,34.1-34.1,34.1c-0.6,0-1.1,0-1.7,0
                                    c-2.4,3.1-4.6,6.2-6.7,9.5l-0.1,0.1c2.7,0.5,5.5,0.8,8.4,0.8c24.6,0,44.5-19.9,44.5-44.5C123.4,44.9,122.3,39.5,120.3,34.4z"/>
                            </g>
                            <g>
                                <g>
                                    <path class="st0" d="M78.9,6.1C54.3,6.1,34.4,26,34.4,50.6c0,0.7,0,1.3,0,2c3.5-2.2,7-4.2,10.6-6.2
                                        c2.1-16.9,16.5-29.9,33.9-29.9c4.1,0,8,0.7,11.6,2l15.3-3.3C98.4,9.5,89,6.1,78.9,6.1z"/>
                                </g>
                            </g>
                            <path class="st0" d="M118.3,18.9c-16.1,1.5-40.9,7.4-50,9.9c-8.9,2.5-16.2,5.1-20.9,6.8c-1.7,3.4-12.6,13-13,17
                                c9.6-5.7,25-13.2,35.8-17.8c9.2-4,27.1-10,30.3-10.9C103.8,22.9,118.3,18.9,118.3,18.9z"/>
                        </g>
                    </g>
                    </svg>
                </div>
                <h1>My Parcel Freight</h1>
            </div>
            
            <div class="loadlink-freight-container">
                <div class="loadlink-freight-header">
                    <div class="loadlink-pagination-controls">
                        <select id="loadlink-per-page" class="loadlink-per-page-select">
                            <option value="10" <?php selected($pagination['per_page'], 10); ?>>10</option>
                            <option value="25" <?php selected($pagination['per_page'], 25); ?>>25</option>
                            <option value="50" <?php selected($pagination['per_page'], 50); ?>>50</option>
                            <option value="100" <?php selected($pagination['per_page'], 100); ?>>100</option>
                        </select>
                        <span class="loadlink-per-page-text">entries per page</span>
                    </div>
                    
                    <div class="loadlink-search-container">
                        <input type="text" id="loadlink-search" class="loadlink-search-input" placeholder="Search..." value="<?php echo esc_attr($pagination['search']); ?>">
                    </div>
                </div>
                
                <div class="loadlink-freight-table-container">
                    <table class="wp-list-table widefat fixed striped loadlink-freight-table">
                        <thead>
                            <tr>
                                <th class="sortable" data-column="freight_id">
                                    <a href="#" class="loadlink-sort-link">
                                        FREIGHT ID
                                        <span class="sort-indicator">↕</span>
                                    </a>
                                </th>
                                <th class="sortable" data-column="name">
                                    <a href="#" class="loadlink-sort-link">
                                        SERVICE
                                        <span class="sort-indicator">↕</span>
                                    </a>
                                </th>
                                <th class="sortable" data-column="date">
                                    <a href="#" class="loadlink-sort-link">
                                        DATE
                                        <span class="sort-indicator">↕</span>
                                    </a>
                                </th>
                                <th class="sortable" data-column="pickup">
                                    <a href="#" class="loadlink-sort-link">
                                        PICKUP
                                        <span class="sort-indicator">↕</span>
                                    </a>
                                </th>
                                <th class="sortable" data-column="delivery">
                                    <a href="#" class="loadlink-sort-link">
                                        DELIVERY
                                        <span class="sort-indicator">↕</span>
                                    </a>
                                </th>
                                <th class="sortable" data-column="order_no">
                                    <a href="#" class="loadlink-sort-link">
                                        ORDER NO.
                                        <span class="sort-indicator">↕</span>
                                    </a>
                                </th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($freight_orders)): ?>
                                <tr>
                                    <td colspan="6" class="loadlink-no-data">No freight orders found</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($freight_orders as $order): ?>
                                    <tr>
                                        <td>
                                            <?php echo esc_html($order['freight_id']); ?>
                                        </td>
                                        <td><?php echo esc_html($order['service']); ?></td>
                                        <td><?php echo esc_html($order['date']); ?></td>
                                        <td><?php echo esc_html($order['pickup']); ?></td>
                                        <td><?php echo esc_html($order['delivery']); ?></td>
                                        <td><?php echo esc_html($order['order_no']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
                
                <?php if ($pagination['total_pages'] > 1): ?>
                    <div class="loadlink-pagination">
                        <?php echo $this->render_pagination($pagination); ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Modal for freight details -->
        <div id="loadlink-freight-modal" class="loadlink-modal" style="display: none;">
            <div class="loadlink-modal-content">
                <div class="loadlink-modal-header">
                    <h2>Freight Details</h2>
                    <span class="loadlink-modal-close">&times;</span>
                </div>
                <div class="loadlink-modal-body">
                    <div id="loadlink-freight-details">
                        <div class="loadlink-loading">Loading...</div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Get freight orders from database
     */
    private function get_freight_orders() {
        global $wpdb;
        
        $per_page = isset($_GET['per_page']) ? intval($_GET['per_page']) : 10;
        $page = isset($_GET['paged']) ? intval($_GET['paged']) : 1;
        $search = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';
        $orderby = isset($_GET['orderby']) ? sanitize_text_field($_GET['orderby']) : 'date';
        $order = isset($_GET['order']) ? sanitize_text_field($_GET['order']) : 'DESC';
        
        $offset = ($page - 1) * $per_page;
        
        // Debug: Let's see what shipping methods exist
        $debug_sql = "SELECT DISTINCT meta_key, meta_value FROM {$wpdb->postmeta} WHERE post_id IN (SELECT ID FROM {$wpdb->posts} WHERE post_type = 'shop_order' AND post_status IN ('wc-processing', 'wc-completed', 'wc-pending', 'wc-on-hold')) AND meta_key LIKE '%shipping%'";
        $debug_results = $wpdb->get_results($debug_sql);
        error_log('Loadlink Debug - Shipping meta keys and values: ' . print_r($debug_results, true));
        
        // Get orders with Loadlink shipping (either with freight data or just Loadlink shipping)
        // Let's try a broader approach to catch all possible Loadlink orders
        $sql = "
            SELECT DISTINCT p.ID as order_id, p.post_date, pm_freight.meta_value as freight_data
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm_freight ON p.ID = pm_freight.post_id AND pm_freight.meta_key = '_loadlink_freight_data'
            WHERE p.post_type = 'shop_order'
            AND p.post_status IN ('wc-processing', 'wc-completed', 'wc-pending', 'wc-on-hold')
            AND (
                pm_freight.meta_key = '_loadlink_freight_data' AND pm_freight.meta_value != ''
                OR p.ID IN (
                    SELECT DISTINCT post_id FROM {$wpdb->postmeta} 
                    WHERE meta_key LIKE '%shipping%' 
                    AND meta_value LIKE '%loadlink%'
                )
                OR p.ID IN (
                    SELECT DISTINCT post_id FROM {$wpdb->postmeta} 
                    WHERE meta_key LIKE '%loadlink%'
                )
            )
        ";
        
        if (!empty($search)) {
            $sql .= $wpdb->prepare(" AND (p.ID LIKE %s OR p.post_title LIKE %s)", '%' . $search . '%', '%' . $search . '%');
        }
        
        // Add ordering
        $allowed_orderby = ['freight_id', 'name', 'date', 'pickup', 'delivery', 'order_no'];
        if (in_array($orderby, $allowed_orderby)) {
            switch ($orderby) {
                case 'freight_id':
                    $sql .= " ORDER BY CAST(JSON_EXTRACT(pm_freight.meta_value, '$.freight_id') AS UNSIGNED) " . $order;
                    break;
                case 'date':
                    $sql .= " ORDER BY p.post_date " . $order;
                    break;
                default:
                    $sql .= " ORDER BY p.post_date DESC";
            }
        } else {
            $sql .= " ORDER BY p.post_date DESC";
        }
        
        $sql .= $wpdb->prepare(" LIMIT %d OFFSET %d", $per_page, $offset);
        
        error_log('Loadlink Debug - Final SQL query: ' . $sql);
        $results = $wpdb->get_results($sql);
        error_log('Loadlink Debug - Query results: ' . print_r($results, true));
        $freight_orders = [];
        
        // Debug: Log the SQL query and results
        Loadlink_Logger::log('Freight orders SQL: ' . $sql, 'debug');
        Loadlink_Logger::log('Found ' . count($results) . ' orders with freight data', 'debug');
        
        // Debug: Check if there are any orders with _loadlink_freight_data meta at all
        $all_freight_meta = $wpdb->get_results("
            SELECT p.ID, pm.meta_value 
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_type = 'shop_order'
            AND pm.meta_key = '_loadlink_freight_data'
        ");
        Loadlink_Logger::log('Total orders with _loadlink_freight_data meta: ' . count($all_freight_meta), 'debug');
        
        foreach ($all_freight_meta as $meta) {
            Loadlink_Logger::log('Order ' . $meta->ID . ' has freight data: ' . $meta->meta_value, 'debug');
        }
        
        // Debug: Let's also check what recent orders exist
        $recent_orders = $wpdb->get_results("
            SELECT p.ID, p.post_title, p.post_status, p.post_date
            FROM {$wpdb->posts} p 
            WHERE p.post_type = 'shop_order' 
            AND p.post_status IN ('wc-processing', 'wc-completed', 'wc-pending', 'wc-on-hold')
            ORDER BY p.post_date DESC 
            LIMIT 10
        ");
        error_log('Loadlink Debug - Recent orders: ' . print_r($recent_orders, true));
        
        // If no results found, let's show recent orders for debugging
        if (empty($results)) {
            error_log('Loadlink Debug - No results found, showing recent orders for debugging');
            $fallback_orders = $wpdb->get_results("
                SELECT p.ID as order_id, p.post_date
                FROM {$wpdb->posts} p 
                WHERE p.post_type = 'shop_order' 
                AND p.post_status IN ('wc-processing', 'wc-completed', 'wc-pending', 'wc-on-hold')
                ORDER BY p.post_date DESC 
                LIMIT 5
            ");
            
            foreach ($fallback_orders as $fallback_order) {
                $order = wc_get_order($fallback_order->order_id);
                if ($order) {
                    $shipping_methods = $order->get_shipping_methods();
                    $has_loadlink = false;
                    $loadlink_method = '';
                    
                    foreach ($shipping_methods as $shipping_method) {
                        $method_id = $shipping_method->get_method_id();
                        $method_title = $shipping_method->get_method_title();
                        error_log('Loadlink Debug - Order ' . $fallback_order->order_id . ' shipping method: ' . $method_id . ' - ' . $method_title);
                        
                        if (strpos($method_id, 'loadlink') !== false || strpos($method_title, 'loadlink') !== false) {
                            $has_loadlink = true;
                            $loadlink_method = $method_title;
                            break;
                        }
                    }
                    
                    if ($has_loadlink) {
                        $freight_orders[] = [
                            'order_id' => $fallback_order->order_id,
                            'freight_id' => 'Pending',
                            'service_name' => $loadlink_method ?: 'Loadlink Shipping',
                            'pickup_address' => $order->get_shipping_address_1() ?: $order->get_billing_address_1(),
                            'delivery_address' => $order->get_shipping_address_1() ?: $order->get_billing_address_1(),
                            'date' => $fallback_order->post_date,
                            'status' => 'pending'
                        ];
                    }
                }
            }
        }
        
        foreach ($results as $result) {
            $freight_data = null;
            if ($result->freight_data) {
                $freight_data = json_decode($result->freight_data, true);
            }
            
            Loadlink_Logger::log('Processing order ' . $result->order_id . ' with freight data: ' . json_encode($freight_data), 'debug');
            
            $order = wc_get_order($result->order_id);
            if (!$order) {
                continue;
            }
            
            // If no freight data, create basic info from order
            if (!$freight_data || !isset($freight_data['freight_id'])) {
                $freight_id = 'pending_' . $result->order_id;
                $freight_data = [
                    'freight_id' => $freight_id,
                    'name' => 'LoadLink Dev',
                    'pickup_address' => [
                        'name' => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
                        'address' => $order->get_billing_address_1(),
                        'suburb' => $order->get_billing_city(),
                        'postcode' => $order->get_billing_postcode(),
                        'state' => $order->get_billing_state()
                    ],
                    'delivery_address' => [
                        'name' => $order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name(),
                        'address' => $order->get_shipping_address_1(),
                        'suburb' => $order->get_shipping_city(),
                        'postcode' => $order->get_shipping_postcode(),
                        'state' => $order->get_shipping_state()
                    ]
                ];
                Loadlink_Logger::log('Created basic freight data for order ' . $result->order_id, 'debug');
            }
            
            $freight_orders[] = [
                'freight_id' => $freight_data['freight_id'],
                'service' => $freight_data['selected_service_name'] ?? 'Unknown',
                'name' => $freight_data['name'] ?? 'LoadLink Dev',
                'date' => date('d/m/Y H:i', strtotime($result->post_date)),
                'pickup' => $this->format_address($freight_data['pickup_address'] ?? []),
                'delivery' => $this->format_address($freight_data['delivery_address'] ?? []),
                'order_no' => $order->get_order_number(),
                'order_id' => $result->order_id,
                'freight_data' => $freight_data
            ];
        }
        
        return $freight_orders;
    }
    
    /**
     * Format address for display
     */
    private function format_address($address) {
        if (empty($address)) {
            return '';
        }
        
        $parts = [];
        if (!empty($address['suburb'])) $parts[] = strtoupper($address['suburb']);
        if (!empty($address['postcode'])) $parts[] = $address['postcode'];
        if (!empty($address['state'])) $parts[] = $address['state'];
        
        return implode(', ', $parts);
    }
    
    /**
     * Get pagination data
     */
    private function get_pagination_data($freight_orders) {
        global $wpdb;
        
        $per_page = isset($_GET['per_page']) ? intval($_GET['per_page']) : 10;
        $page = isset($_GET['paged']) ? intval($_GET['paged']) : 1;
        $search = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';
        
        // Get total count
        $count_sql = "
            SELECT COUNT(DISTINCT p.ID)
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm_freight ON p.ID = pm_freight.post_id AND pm_freight.meta_key = '_loadlink_freight_data'
            INNER JOIN {$wpdb->postmeta} pm_shipping ON p.ID = pm_shipping.post_id
            WHERE p.post_type = 'shop_order'
            AND p.post_status IN ('wc-processing', 'wc-completed', 'wc-pending', 'wc-on-hold')
            AND (
                (pm_shipping.meta_key = '_shipping_method' AND pm_shipping.meta_value LIKE '%loadlink%')
                OR (pm_shipping.meta_key = '_shipping_methods' AND pm_shipping.meta_value LIKE '%loadlink%')
                OR (pm_freight.meta_key = '_loadlink_freight_data' AND pm_freight.meta_value != '')
            )
        ";
        
        if (!empty($search)) {
            $count_sql .= $wpdb->prepare(" AND (p.ID LIKE %s OR p.post_title LIKE %s)", '%' . $search . '%', '%' . $search . '%');
        }
        
        $total_items = $wpdb->get_var($count_sql);
        $total_pages = ceil($total_items / $per_page);
        
        return [
            'per_page' => $per_page,
            'page' => $page,
            'total_items' => $total_items,
            'total_pages' => $total_pages,
            'search' => $search
        ];
    }
    
    /**
     * Render pagination links
     */
    private function render_pagination($pagination) {
        $current_page = $pagination['page'];
        $total_pages = $pagination['total_pages'];
        $base_url = remove_query_arg(['paged', 'per_page', 's']);
        
        $output = '<div class="loadlink-pagination-links">';
        
        // Previous page
        if ($current_page > 1) {
            $prev_url = add_query_arg('paged', $current_page - 1, $base_url);
            $output .= '<a href="' . esc_url($prev_url) . '" class="loadlink-page-link">&laquo; Previous</a>';
        }
        
        // Page numbers
        $start = max(1, $current_page - 2);
        $end = min($total_pages, $current_page + 2);
        
        for ($i = $start; $i <= $end; $i++) {
            if ($i == $current_page) {
                $output .= '<span class="loadlink-page-current">' . $i . '</span>';
            } else {
                $page_url = add_query_arg('paged', $i, $base_url);
                $output .= '<a href="' . esc_url($page_url) . '" class="loadlink-page-link">' . $i . '</a>';
            }
        }
        
        // Next page
        if ($current_page < $total_pages) {
            $next_url = add_query_arg('paged', $current_page + 1, $base_url);
            $output .= '<a href="' . esc_url($next_url) . '" class="loadlink-page-link">Next &raquo;</a>';
        }
        
        $output .= '</div>';
        
        return $output;
    }
    
    /**
     * AJAX handler for freight details
     */
    public function ajax_get_freight_details() {
        check_ajax_referer('loadlink_admin_nonce', 'nonce');
        
        $freight_id = intval($_POST['freight_id']);
        $order_id = intval($_POST['order_id']);
        
        if (!$freight_id || !$order_id) {
            wp_send_json_error('Invalid parameters');
        }
        
        $order = wc_get_order($order_id);
        if (!$order) {
            wp_send_json_error('Order not found');
        }
        
        $freight_data = $order->get_meta('_loadlink_freight_data');
        if (!$freight_data || $freight_data['freight_id'] != $freight_id) {
            wp_send_json_error('Freight data not found');
        }
        
        $details = $this->format_freight_details($freight_data, $order);
        
        wp_send_json_success($details);
    }
    
    /**
     * Format freight details for display
     */
    private function format_freight_details($freight_data, $order) {
        $html = '<div class="loadlink-freight-details">';
        
        // Basic info
        $html .= '<div class="loadlink-detail-section">';
        $html .= '<h3>Freight Information</h3>';
        $html .= '<table class="loadlink-details-table">';
        $html .= '<tr><td><strong>Freight ID:</strong></td><td>' . esc_html($freight_data['freight_id']) . '</td></tr>';
        $html .= '<tr><td><strong>Order Number:</strong></td><td>' . esc_html($order->get_order_number()) . '</td></tr>';
        $html .= '<tr><td><strong>Order Date:</strong></td><td>' . esc_html($order->get_date_created()->date('d/m/Y H:i')) . '</td></tr>';
        $html .= '<tr><td><strong>Status:</strong></td><td>' . esc_html(wc_get_order_status_name($order->get_status())) . '</td></tr>';
        $html .= '</table>';
        $html .= '</div>';
        
        // Pickup address
        if (!empty($freight_data['pickup_address'])) {
            $html .= '<div class="loadlink-detail-section">';
            $html .= '<h3>Pickup Address</h3>';
            $html .= '<div class="loadlink-address">';
            $html .= '<p><strong>' . esc_html($freight_data['pickup_address']['name'] ?? '') . '</strong></p>';
            $html .= '<p>' . esc_html($freight_data['pickup_address']['address'] ?? '') . '</p>';
            $html .= '<p>' . esc_html($freight_data['pickup_address']['suburb'] ?? '') . ' ' . esc_html($freight_data['pickup_address']['postcode'] ?? '') . '</p>';
            $html .= '<p>' . esc_html($freight_data['pickup_address']['state'] ?? '') . '</p>';
            $html .= '</div>';
            $html .= '</div>';
        }
        
        // Delivery address
        if (!empty($freight_data['delivery_address'])) {
            $html .= '<div class="loadlink-detail-section">';
            $html .= '<h3>Delivery Address</h3>';
            $html .= '<div class="loadlink-address">';
            $html .= '<p><strong>' . esc_html($freight_data['delivery_address']['name'] ?? '') . '</strong></p>';
            $html .= '<p>' . esc_html($freight_data['delivery_address']['address'] ?? '') . '</p>';
            $html .= '<p>' . esc_html($freight_data['delivery_address']['suburb'] ?? '') . ' ' . esc_html($freight_data['delivery_address']['postcode'] ?? '') . '</p>';
            $html .= '<p>' . esc_html($freight_data['delivery_address']['state'] ?? '') . '</p>';
            $html .= '</div>';
            $html .= '</div>';
        }
        
        // Service details
        if (!empty($freight_data['selected_service_quote_id'])) {
            $html .= '<div class="loadlink-detail-section">';
            $html .= '<h3>Service Details</h3>';
            $html .= '<table class="loadlink-details-table">';
            $html .= '<tr><td><strong>Service:</strong></td><td>' . esc_html($freight_data['selected_service_name'] ?? 'Unknown') . '</td></tr>';
            $html .= '<tr><td><strong>Quote ID:</strong></td><td>' . esc_html($freight_data['selected_service_quote_id']) . '</td></tr>';
            $html .= '</table>';
            $html .= '</div>';
        }
        
        $html .= '</div>';
        
        return $html;
    }
    
    /**
     * AJAX handler for creating test freight data
     */
    public function ajax_create_test_freight() {
        check_ajax_referer('loadlink_admin_nonce', 'nonce');
        
        // Get recent orders with Loadlink shipping
        global $wpdb;
        
        $orders = $wpdb->get_results("
            SELECT p.ID, p.post_date
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_type = 'shop_order'
            AND p.post_status IN ('wc-processing', 'wc-completed', 'wc-pending', 'wc-on-hold')
            AND pm.meta_key = '_shipping_method'
            AND pm.meta_value LIKE '%loadlink%'
            ORDER BY p.post_date DESC
            LIMIT 5
        ");
        
        $created_count = 0;
        
        foreach ($orders as $order_data) {
            $order = wc_get_order($order_data->ID);
            if (!$order) continue;
            
            // Check if freight data already exists
            $existing_freight = $order->get_meta('_loadlink_freight_data');
            if (!empty($existing_freight)) {
                continue; // Skip if already has freight data
            }
            
            // Create test freight data
            $freight_data = [
                'freight_id' => 'test_' . $order_data->ID . '_' . time(),
                'name' => 'LoadLink Dev',
                'pickup_address' => [
                    'name' => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
                    'address' => $order->get_billing_address_1(),
                    'suburb' => $order->get_billing_city(),
                    'postcode' => $order->get_billing_postcode(),
                    'state' => $order->get_billing_state()
                ],
                'delivery_address' => [
                    'name' => $order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name(),
                    'address' => $order->get_shipping_address_1(),
                    'suburb' => $order->get_shipping_city(),
                    'postcode' => $order->get_shipping_postcode(),
                    'state' => $order->get_shipping_state()
                ],
                'service_quote_ids' => [
                    'road express' => 'test_quote_1',
                    'overnight express' => 'test_quote_2',
                    'technology express' => 'test_quote_3'
                ],
                'selected_service_quote_id' => 'test_quote_1',
                'selected_service_name' => 'road express'
            ];
            
            // Save freight data
            $order->update_meta_data('_loadlink_freight_data', $freight_data);
            $order->update_meta_data('_loadlink_freight_id', $freight_data['freight_id']);
            $order->update_meta_data('_loadlink_service_quote_id', $freight_data['selected_service_quote_id']);
            $order->update_meta_data('_loadlink_selected_service_name', $freight_data['selected_service_name']);
            $order->save();
            
            $created_count++;
        }
        
        wp_send_json_success([
            'message' => "Created test freight data for {$created_count} orders",
            'count' => $created_count
        ]);
    }
    
    /**
     * AJAX handler for creating freight data for a specific order
     */
    public function ajax_create_freight_for_order() {
        check_ajax_referer('loadlink_admin_nonce', 'nonce');
        
        $order_id = intval($_POST['order_id']);
        if (!$order_id) {
            wp_send_json_error('Invalid order ID');
        }
        
        $order = wc_get_order($order_id);
        if (!$order) {
            wp_send_json_error('Order not found');
        }
        
        // Check if freight data already exists
        $existing_freight = $order->get_meta('_loadlink_freight_data');
        if (!empty($existing_freight)) {
            wp_send_json_error('Freight data already exists for this order');
        }
        
        // Create freight data for this specific order
        $freight_data = [
            'freight_id' => 'order_' . $order_id . '_' . time(),
            'name' => 'LoadLink Dev',
            'pickup_address' => [
                'name' => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
                'address' => $order->get_billing_address_1(),
                'suburb' => $order->get_billing_city(),
                'postcode' => $order->get_billing_postcode(),
                'state' => $order->get_billing_state()
            ],
            'delivery_address' => [
                'name' => $order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name(),
                'address' => $order->get_shipping_address_1(),
                'suburb' => $order->get_shipping_city(),
                'postcode' => $order->get_shipping_postcode(),
                'state' => $order->get_shipping_state()
            ],
            'service_quote_ids' => [
                'road express' => 'order_quote_1_' . $order_id,
                'overnight express' => 'order_quote_2_' . $order_id,
                'technology express' => 'order_quote_3_' . $order_id
            ],
            'selected_service_quote_id' => 'order_quote_1_' . $order_id,
            'selected_service_name' => 'road express'
        ];
        
        // Save freight data
        $order->update_meta_data('_loadlink_freight_data', $freight_data);
        $order->update_meta_data('_loadlink_freight_id', $freight_data['freight_id']);
        $order->update_meta_data('_loadlink_service_quote_id', $freight_data['selected_service_quote_id']);
        $order->update_meta_data('_loadlink_selected_service_name', $freight_data['selected_service_name']);
        $order->save();
        
        wp_send_json_success([
            'message' => 'Freight data created successfully',
            'freight_id' => $freight_data['freight_id']
        ]);
    }
}
