<?php
/**
 * Debug script to identify why order confirmation endpoint is not being called
 * Place this file in your WordPress root directory and access via browser
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    // Load WordPress
    require_once('wp-config.php');
    require_once('wp-load.php');
}

// Check if WooCommerce is active
if (!class_exists('WooCommerce')) {
    die('WooCommerce is not active');
}

echo "<h1>Loadlink Order Confirmation Debug</h1>";

// 1. Check if Loadlink plugin is active
echo "<h2>1. Plugin Status</h2>";
if (class_exists('Loadlink_Order_Confirmation')) {
    echo "✅ Loadlink_Order_Confirmation class exists<br>";
} else {
    echo "❌ Loadlink_Order_Confirmation class NOT found<br>";
}

if (class_exists('Loadlink_Plugin')) {
    echo "✅ Loadlink_Plugin class exists<br>";
} else {
    echo "❌ Loadlink_Plugin class NOT found<br>";
}

// 2. Check if hooks are registered
echo "<h2>2. Hook Registration</h2>";
global $wp_filter;

$hooks_to_check = [
    'woocommerce_checkout_order_processed',
    'woocommerce_order_status_processing', 
    'woocommerce_order_status_completed'
];

foreach ($hooks_to_check as $hook) {
    if (isset($wp_filter[$hook])) {
        echo "✅ Hook '$hook' is registered<br>";
        
        // Check for Loadlink callbacks
        $callbacks = $wp_filter[$hook]->callbacks;
        $found_loadlink = false;
        
        foreach ($callbacks as $priority => $hooks) {
            foreach ($hooks as $callback) {
                if (is_array($callback['function']) && 
                    is_string($callback['function'][0]) && 
                    strpos($callback['function'][0], 'Loadlink') !== false) {
                    echo "&nbsp;&nbsp;✅ Found Loadlink callback: " . $callback['function'][0] . "::" . $callback['function'][1] . " (priority: $priority)<br>";
                    $found_loadlink = true;
                }
            }
        }
        
        if (!$found_loadlink) {
            echo "&nbsp;&nbsp;❌ No Loadlink callbacks found for '$hook'<br>";
        }
    } else {
        echo "❌ Hook '$hook' is NOT registered<br>";
    }
}

// 3. Check recent orders for Loadlink shipping
echo "<h2>3. Recent Orders Analysis</h2>";
$recent_orders = wc_get_orders([
    'limit' => 10,
    'orderby' => 'date',
    'order' => 'DESC'
]);

if (empty($recent_orders)) {
    echo "No recent orders found<br>";
} else {
    foreach ($recent_orders as $order) {
        echo "<h3>Order #" . $order->get_id() . " - " . $order->get_status() . "</h3>";
        
        // Check shipping methods
        $shipping_methods = $order->get_shipping_methods();
        $has_loadlink = false;
        
        foreach ($shipping_methods as $shipping_method) {
            $method_id = $shipping_method->get_method_id();
            echo "Shipping Method: $method_id<br>";
            
            if (strpos($method_id, 'loadlink') !== false) {
                $has_loadlink = true;
                echo "✅ This order uses Loadlink shipping<br>";
            }
        }
        
        if (!$has_loadlink) {
            echo "❌ This order does NOT use Loadlink shipping<br>";
        }
        
        // Check for Loadlink meta data
        $freight_id = $order->get_meta('_loadlink_freight_id');
        $service_quote_id = $order->get_meta('_loadlink_service_quote_id');
        $confirmation_status = $order->get_meta('_loadlink_confirmation_status');
        
        echo "Freight ID: " . ($freight_id ?: 'Not set') . "<br>";
        echo "Service Quote ID: " . ($service_quote_id ?: 'Not set') . "<br>";
        echo "Confirmation Status: " . ($confirmation_status ?: 'Not set') . "<br>";
        
        if ($freight_id && $service_quote_id) {
            echo "✅ Order has required Loadlink data<br>";
        } else {
            echo "❌ Order missing required Loadlink data<br>";
        }
        
        echo "<hr>";
    }
}

// 4. Check WooCommerce logs
echo "<h2>4. WooCommerce Logs</h2>";
$log_dir = WC_LOG_DIR;
if (is_dir($log_dir)) {
    $log_files = glob($log_dir . '*.log');
    if (!empty($log_files)) {
        echo "Found " . count($log_files) . " log files<br>";
        
        // Look for Loadlink logs
        foreach ($log_files as $log_file) {
            if (strpos($log_file, 'loadlink') !== false) {
                echo "📄 Loadlink log file: " . basename($log_file) . "<br>";
                
                // Show last 20 lines
                $lines = file($log_file);
                $last_lines = array_slice($lines, -20);
                echo "<pre style='background: #f0f0f0; padding: 10px; max-height: 300px; overflow-y: auto;'>";
                echo htmlspecialchars(implode('', $last_lines));
                echo "</pre>";
            }
        }
    } else {
        echo "No log files found<br>";
    }
} else {
    echo "Log directory not found: $log_dir<br>";
}

// 5. Test session data
echo "<h2>5. Session Data Test</h2>";
if (function_exists('WC') && WC()->session) {
    echo "✅ WooCommerce session is available<br>";
    
    $freight_data = WC()->session->get('loadlink_freight_data');
    $selected_service_quote_id = WC()->session->get('loadlink_selected_service_quote_id');
    $selected_service_name = WC()->session->get('loadlink_selected_service_name');
    
    echo "Freight data in session: " . ($freight_data ? 'Present' : 'Not present') . "<br>";
    echo "Selected service quote ID: " . ($selected_service_quote_id ?: 'Not set') . "<br>";
    echo "Selected service name: " . ($selected_service_name ?: 'Not set') . "<br>";
    
    if ($freight_data) {
        echo "<pre style='background: #f0f0f0; padding: 10px;'>";
        echo htmlspecialchars(print_r($freight_data, true));
        echo "</pre>";
    }
} else {
    echo "❌ WooCommerce session not available<br>";
}

// 6. Check if order confirmation class is properly initialized
echo "<h2>6. Order Confirmation Initialization</h2>";
if (class_exists('Loadlink_Order_Confirmation')) {
    echo "✅ Loadlink_Order_Confirmation class exists<br>";
    
    // Check if init method exists
    if (method_exists('Loadlink_Order_Confirmation', 'init')) {
        echo "✅ init() method exists<br>";
    } else {
        echo "❌ init() method NOT found<br>";
    }
    
    // Check if process_order_confirmation method exists
    if (method_exists('Loadlink_Order_Confirmation', 'process_order_confirmation')) {
        echo "✅ process_order_confirmation() method exists<br>";
    } else {
        echo "❌ process_order_confirmation() method NOT found<br>";
    }
} else {
    echo "❌ Loadlink_Order_Confirmation class NOT found<br>";
}

// 7. Check WordPress and WooCommerce versions
echo "<h2>7. System Information</h2>";
echo "WordPress Version: " . get_bloginfo('version') . "<br>";
echo "WooCommerce Version: " . WC()->version . "<br>";
echo "PHP Version: " . PHP_VERSION . "<br>";

// 8. Check for potential conflicts
echo "<h2>8. Potential Issues</h2>";

// Check if HPOS is enabled
if (class_exists('Automattic\WooCommerce\Utilities\OrderUtil')) {
    if (\Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled()) {
        echo "⚠️ HPOS (High-Performance Order Storage) is enabled - this might affect order processing<br>";
    } else {
        echo "✅ HPOS is not enabled<br>";
    }
}

// Check for other shipping plugins
$active_plugins = get_option('active_plugins');
$shipping_plugins = [];
foreach ($active_plugins as $plugin) {
    if (strpos($plugin, 'shipping') !== false || strpos($plugin, 'woocommerce') !== false) {
        $shipping_plugins[] = $plugin;
    }
}

if (count($shipping_plugins) > 1) {
    echo "⚠️ Multiple shipping-related plugins detected:<br>";
    foreach ($shipping_plugins as $plugin) {
        echo "&nbsp;&nbsp;- $plugin<br>";
    }
} else {
    echo "✅ No conflicting shipping plugins detected<br>";
}

echo "<h2>Debug Complete</h2>";
echo "<p>If you're still experiencing issues, please check:</p>";
echo "<ul>";
echo "<li>WooCommerce logs for any error messages</li>";
echo "<li>WordPress debug logs</li>";
echo "<li>Server error logs</li>";
echo "<li>Test with a simple order using Loadlink shipping</li>";
echo "</ul>";
?>

