(function(){
	// Debouncing mechanism to prevent rapid API calls
	let debounceTimer = null;
	const DEBOUNCE_DELAY = 1000; // 1 second delay
	
	function findShippingContainer(){
		// Prefer Blocks address form wrapper
		return document.querySelector('.wc-block-components-address-form, .wc-block-components-shipping-address, .woocommerce-shipping-fields .woocommerce-shipping-fields__field-wrapper, .woocommerce-shipping-fields');
	}

	function attachListener(select) {
		if (select.dataset.loadlinkListener === 'true') return;
		
		select.dataset.loadlinkListener = 'true';
		select.addEventListener('change', function(){
			if(!window.loadlinkAjax) return;
			
			// Debounce the API call to prevent rapid successive requests
			clearTimeout(debounceTimer);
			debounceTimer = setTimeout(function() {
				var fd = new FormData();
				fd.append('action','loadlink_set_dropoff_type');
				fd.append('nonce', window.loadlinkAjax.nonce);
				fd.append('value', select.value);
				fetch(window.loadlinkAjax.ajaxUrl, { method: 'POST', body: fd })
				.then(function() {
					// Trigger WooCommerce update_checkout to refresh rates if jQuery is available
					if (typeof jQuery !== 'undefined' && typeof jQuery(document.body).trigger === 'function') {
						jQuery(document.body).trigger('update_checkout');
					}
				});
			}, DEBOUNCE_DELAY);
		});
		
		// Ensure initial session value matches (only do this for one field to avoid race conditions)
		if (window.loadlinkAjax && window.loadlinkAjax.sessionValue && window.loadlinkAjax.sessionValue !== select.value) {
			// Only update if this is the active field (visible)
			if (select.offsetParent !== null) {
				clearTimeout(debounceTimer);
				debounceTimer = setTimeout(function() {
					var fdInit = new FormData();
					fdInit.append('action','loadlink_set_dropoff_type');
					fdInit.append('nonce', window.loadlinkAjax.nonce);
					fdInit.append('value', select.value);
					fetch(window.loadlinkAjax.ajaxUrl, { method: 'POST', body: fdInit });
				}, DEBOUNCE_DELAY);
			}
		}
	}

	function insertField(container){
		// Check for Billing Field (added via PHP in Classic, or potentially injected)
		var billing = document.getElementById('billing_dropoff_building_type');
		if (billing) attachListener(billing);

		// Check for Shipping Field
		var existing = document.getElementById('shipping_dropoff_building_type');
		if (existing) {
			attachListener(existing);
			return true;
		}

		var wrapper = document.createElement('div');
		wrapper.className = 'form-row form-row-wide';
		var label = document.createElement('label');
		label.setAttribute('for','shipping_dropoff_building_type');
		label.textContent = 'Dropoff Building Type';
		var select = document.createElement('select');
		select.id = 'shipping_dropoff_building_type';
		select.name = 'shipping_dropoff_building_type';
		var opt1 = document.createElement('option');
		opt1.value = 'Residential'; opt1.text = 'Residential';
		var opt2 = document.createElement('option');
		opt2.value = 'Commercial'; opt2.text = 'Commercial';
		select.appendChild(opt1); select.appendChild(opt2);
		if (window.loadlinkAjax && window.loadlinkAjax.default){
			select.value = window.loadlinkAjax.default;
		}
		wrapper.appendChild(label);
		wrapper.appendChild(select);
		attachListener(select);

		// Prefer to insert directly below the Last name row (within the same container)
		var last = document.querySelector('input[name="shipping-last_name"], #shipping-last_name, input[autocomplete="family-name"]');
		var first = document.querySelector('input[name="shipping-first_name"], #shipping-first_name, input[autocomplete="given-name"]');
		var anchor = null;
		if (last) {
			anchor = last.closest('.wc-block-components-text-input, .components-base-control, .form-row, div');
		} else if (first) {
			anchor = first.closest('.wc-block-components-text-input, .components-base-control, .form-row, div');
		}
		if (anchor) {
			anchor.insertAdjacentElement('afterend', wrapper);
		} else {
			// Fallback: after postcode or at end
			var postcode = container.querySelector('input[autocomplete="postal-code"], input[name*="postcode" i], #shipping-postcode');
			if (postcode && postcode.parentElement){
				postcode.parentElement.insertAdjacentElement('afterend', wrapper);
			} else {
				container.appendChild(wrapper);
			}
		}

		return true;
	}

	function ensureFieldWithRetries(){
		var attempts = 0;
		var maxAttempts = 30;
		var timer = setInterval(function(){
			var container = findShippingContainer();
			if (container && insertField(container)){
				clearInterval(timer);
			}
			if (++attempts >= maxAttempts){
				clearInterval(timer);
			}
		}, 200);
	}

	function init(){
		ensureFieldWithRetries();
		// Also observe for later re-renders
		var obs = new MutationObserver(function(){ ensureFieldWithRetries(); });
		obs.observe(document.body, { childList: true, subtree: true });
	}

	// Blocks checkout order confirmation handling
	function handleBlocksCheckoutConfirmation() {
		console.log('Loadlink: Setting up Blocks checkout confirmation handlers');
		
		// Listen for Blocks checkout completion events
		document.addEventListener('checkout_error', function(event) {
			console.log('Loadlink: Checkout error detected', event);
		});
		
		// Listen for successful checkout
		document.addEventListener('checkout_success', function(event) {
			console.log('Loadlink: Checkout success detected', event);
			// Trigger order confirmation for Blocks checkout
			triggerBlocksOrderConfirmation();
		});
		
		// Listen for order received page
		if (window.location.href.includes('order-received') || window.location.href.includes('checkout/order-received')) {
			console.log('Loadlink: Order received page detected');
			triggerBlocksOrderConfirmation();
		}
		
		// Additional event listeners for different checkout implementations
		document.addEventListener('woocommerce_checkout_place_order', function(event) {
			console.log('Loadlink: WooCommerce checkout place order event', event);
		});
		
		document.addEventListener('woocommerce_checkout_order_processed', function(event) {
			console.log('Loadlink: WooCommerce checkout order processed event', event);
		});
		
		// Listen for page changes (for SPA-like behavior)
		window.addEventListener('popstate', function(event) {
			console.log('Loadlink: Page state changed', event);
			if (window.location.href.includes('order-received')) {
				triggerBlocksOrderConfirmation();
			}
		});
		
		// Fallback: Check periodically if we're on order received page
		let checkCount = 0;
		const maxChecks = 20; // Check for 10 seconds
		const checkInterval = setInterval(function() {
			checkCount++;
			if (window.location.href.includes('order-received') || window.location.href.includes('checkout/order-received')) {
				console.log('Loadlink: Order received page detected via polling');
				triggerBlocksOrderConfirmation();
				clearInterval(checkInterval);
			} else if (checkCount >= maxChecks) {
				console.log('Loadlink: Stopping order received page polling');
				clearInterval(checkInterval);
			}
		}, 500);
	}
	
	function triggerBlocksOrderConfirmation() {
		console.log('Loadlink: triggerBlocksOrderConfirmation called');
		
		// Extract order ID from URL or page content
		let orderId = null;
		
		// Try to get order ID from URL
		const urlMatch = window.location.href.match(/order-received\/(\d+)/);
		if (urlMatch) {
			orderId = urlMatch[1];
			console.log('Loadlink: Found order ID from URL:', orderId);
		}
		
		// Try to get order ID from page content
		if (!orderId) {
			const orderElement = document.querySelector('[data-order-id], .order-number, .woocommerce-order-overview__order strong, .woocommerce-order-overview__order .order-number');
			if (orderElement) {
				const orderText = orderElement.textContent || orderElement.innerText;
				const orderMatch = orderText.match(/\d+/);
				if (orderMatch) {
					orderId = orderMatch[0];
					console.log('Loadlink: Found order ID from page content:', orderId);
				}
			}
		}
		
		// Additional fallback - look for order ID in various places
		if (!orderId) {
			const orderElements = document.querySelectorAll('[class*="order"], [id*="order"], [data-order]');
			for (let element of orderElements) {
				const text = element.textContent || element.innerText;
				const match = text.match(/order[:\s#]*(\d+)/i);
				if (match) {
					orderId = match[1];
					console.log('Loadlink: Found order ID from fallback search:', orderId);
					break;
				}
			}
		}
		
		if (orderId && window.loadlinkAjax) {
			console.log('Loadlink: Triggering order confirmation for order:', orderId);
			console.log('Loadlink: AJAX URL:', window.loadlinkAjax.ajaxUrl);
			console.log('Loadlink: Nonce available:', !!window.loadlinkAjax.confirmNonce);
			
			// Use the confirm nonce for order confirmation
			const nonce = window.loadlinkAjax.confirmNonce || window.loadlinkAjax.nonce;
			
			const formData = new FormData();
			formData.append('action', 'loadlink_confirm_order');
			formData.append('order_id', orderId);
			formData.append('nonce', nonce);
			
			fetch(window.loadlinkAjax.ajaxUrl, {
				method: 'POST',
				body: formData
			}).then(response => {
				console.log('Loadlink: AJAX response status:', response.status);
				return response.json();
			})
			.then(data => {
				if (data.success) {
					console.log('Loadlink: Order confirmation successful', data);
				} else {
					console.error('Loadlink: Order confirmation failed', data);
				}
			})
			.catch(error => {
				console.error('Loadlink: Order confirmation error', error);
			});
		} else {
			console.log('Loadlink: Cannot trigger confirmation - missing orderId or loadlinkAjax data');
			console.log('Loadlink: OrderId:', orderId);
			console.log('Loadlink: loadlinkAjax available:', !!window.loadlinkAjax);
		}
	}
	
	// Enhanced initialization for Blocks checkout
	function initBlocksCheckout() {
		console.log('Loadlink: Initializing Blocks checkout detection');
		
		// Check if we're on a Blocks checkout page
		const isBlocksCheckout = document.querySelector('.wc-block-checkout, .wp-block-woocommerce-checkout, .wc-block-components-checkout, .wc-block-checkout-form, .wp-block-woocommerce-checkout-form');
		
		if (isBlocksCheckout) {
			console.log('Loadlink: Blocks checkout detected');
			handleBlocksCheckoutConfirmation();
		} else {
			console.log('Loadlink: No Blocks checkout detected, checking for other checkout types');
		}
		
		// Also check for order received page
		if (window.location.href.includes('order-received') || window.location.href.includes('checkout/order-received')) {
			console.log('Loadlink: Order received page detected');
			handleBlocksCheckoutConfirmation();
		}
		
		// Additional detection for different themes
		const checkoutForm = document.querySelector('form.checkout, .woocommerce-checkout, .checkout');
		if (checkoutForm) {
			console.log('Loadlink: Checkout form detected, setting up confirmation handling');
			handleBlocksCheckoutConfirmation();
		}
		
		// Add manual trigger button for debugging (only in admin or if debug mode)
		if (window.location.href.includes('order-received') && (window.location.search.includes('debug=1') || window.location.search.includes('loadlink_debug=1'))) {
			const debugButton = document.createElement('button');
			debugButton.textContent = 'Loadlink: Manual Order Confirmation';
			debugButton.style.cssText = 'position: fixed; top: 10px; right: 10px; z-index: 9999; background: #0073aa; color: white; padding: 10px; border: none; border-radius: 4px; cursor: pointer;';
			debugButton.onclick = function() {
				console.log('Loadlink: Manual confirmation triggered');
				triggerBlocksOrderConfirmation();
			};
			document.body.appendChild(debugButton);
			console.log('Loadlink: Debug button added - add ?debug=1 to URL to see it');
		}
	}

	if(document.readyState === 'loading'){
		document.addEventListener('DOMContentLoaded', function() {
			init();
			initBlocksCheckout();
		});
	}else{ 
		init(); 
		initBlocksCheckout();
	}
})();


