# Order Confirmation Payload Format Update

## Overview

Updated the order confirmation payload format to remove the `freight_id` field and match the reference format provided.

## Changes Made

### 1. Updated Payload Structure

**Before:**
```json
{
    "freight_id": 1093,
    "freight_service_quote_id": 1208,
    "external_order_id": "98-1761193339",
    "external_order_no": "98-1761193339",
    // ... address fields
}
```

**After:**
```json
{
    "freight_service_quote_id": 1208,
    "external_order_id": "98-1761193339",
    "external_order_no": "98-1761193339",
    // ... address fields
}
```

### 2. Method Signature Updates

#### Loadlink_Client::confirm_freight_order()
**Before:**
```php
public function confirm_freight_order($freight_id, $service_quote_id, $order_data)
```

**After:**
```php
public function confirm_freight_order($service_quote_id, $order_data)
```

#### Loadlink_Client::build_confirmation_payload()
**Before:**
```php
private function build_confirmation_payload($freight_id, $service_quote_id, $order_data)
```

**After:**
```php
private function build_confirmation_payload($service_quote_id, $order_data)
```

#### Loadlink_Client::make_confirmation_request()
**Before:**
```php
private function make_confirmation_request($payload, $freight_id)
```

**After:**
```php
private function make_confirmation_request($payload)
```

#### Loadlink_Order_Confirmation::confirm_freight_order()
**Before:**
```php
private static function confirm_freight_order($freight_id, $service_quote_id, $order_data, $order)
```

**After:**
```php
private static function confirm_freight_order($service_quote_id, $order_data, $order)
```

### 3. Updated Method Calls

#### In Loadlink_Order_Confirmation::process_order_confirmation()
**Before:**
```php
// Get freight data from session or order meta
$freight_id = self::get_freight_id_from_order($order);
$service_quote_id = self::get_service_quote_id_from_order($order);

if (!$freight_id || !$service_quote_id) {
    Loadlink_Logger::log('Order ' . $order_id . ' missing freight_id or service_quote_id, skipping confirmation', 'warning');
    return;
}

// Confirm the freight order
self::confirm_freight_order($freight_id, $service_quote_id, $order_data, $order);
```

**After:**
```php
// Get service quote ID from order meta
$service_quote_id = self::get_service_quote_id_from_order($order);

if (!$service_quote_id) {
    Loadlink_Logger::log('Order ' . $order_id . ' missing service_quote_id, skipping confirmation', 'warning');
    return;
}

// Confirm the freight order
self::confirm_freight_order($service_quote_id, $order_data, $order);
```

### 4. Updated Logging

#### Removed freight_id from debug logs:
**Before:**
```php
Loadlink_Logger::log('Freight ID: ' . $freight_id, 'debug');
Loadlink_Logger::log('freight_id: ' . var_export($payload['freight_id'], true), 'debug');
```

**After:**
```php
Loadlink_Logger::log('Service Quote ID: ' . $service_quote_id, 'debug');
Loadlink_Logger::log('freight_service_quote_id: ' . var_export($payload['freight_service_quote_id'], true), 'debug');
```

#### Updated order notes:
**Before:**
```php
$order->add_order_note('Loadlink freight order confirmed successfully. Freight ID: ' . $freight_id . ', Service Quote ID: ' . $service_quote_id);
```

**After:**
```php
$order->add_order_note('Loadlink freight order confirmed successfully. Service Quote ID: ' . $service_quote_id);
```

### 5. Payload Validation

The payload now matches the reference format exactly:

```json
{
    "freight_service_quote_id": 1070,
    "external_order_id": "PD001",
    "external_order_no": "OD001",
    "pickup_state": "QLD",
    "pickup_suburb": "SOUTHPORT",
    "pickup_postcode": "4215",
    "pickup_address1": "36 Johnston St",
    "pickup_address2": "8 kaertl str",
    "pickup_address3": "",
    "pickup_company_name": "",
    "pickup_company_contact_name": "Peter Web",
    "pickup_company_email": "peter.web@test.com",
    "pickup_company_phone_area_code": "44",
    "pickup_company_phone": "0401010101",
    "dropoff_state": "QLD",
    "dropoff_suburb": "PARADISE POINT",
    "dropoff_postcode": "4216",
    "dropoff_address1": "1779 Murbko Rd",
    "dropoff_address2": null,
    "dropoff_address3": "",
    "dropoff_company_name": "",
    "dropoff_company_contact_name": "Church State",
    "dropoff_company_email": "church@test.com",
    "dropoff_company_phone_area_code": "44",
    "dropoff_company_phone": "0403030303"
}
```

## Files Modified

1. **`includes/class-loadlink-client.php`**
   - Updated `confirm_freight_order()` method signature
   - Updated `build_confirmation_payload()` method signature
   - Updated `make_confirmation_request()` method signature
   - Removed `freight_id` from payload array
   - Updated logging to remove freight_id references
   - Updated validation logging

2. **`includes/class-loadlink-order-confirmation.php`**
   - Updated `confirm_freight_order()` method signature
   - Updated `process_order_confirmation()` to remove freight_id requirement
   - Updated method calls to remove freight_id parameter
   - Updated order notes to remove freight_id reference

## Testing

### Test Script
Created `test-payload-format.php` to verify:
- Payload structure matches reference format
- All required fields are present
- freight_id is correctly removed
- Data types are correct
- Method signatures are updated

### Usage
```bash
# Access the test script
https://your-site.com/test-payload-format.php
```

## Benefits

1. **Simplified Payload**: Removed unnecessary freight_id field
2. **API Compatibility**: Matches the expected API format exactly
3. **Cleaner Code**: Reduced parameter passing and complexity
4. **Better Logging**: More focused logging without freight_id noise
5. **Reference Compliance**: Exactly matches the provided reference format

## Backward Compatibility

⚠️ **Breaking Change**: This update changes method signatures and removes the freight_id parameter. Any custom code that calls these methods directly will need to be updated.

### Migration Guide

If you have custom code calling these methods:

**Before:**
```php
$client->confirm_freight_order($freight_id, $service_quote_id, $order_data);
```

**After:**
```php
$client->confirm_freight_order($service_quote_id, $order_data);
```

## Verification

To verify the changes are working correctly:

1. **Run the test script**: `test-payload-format.php`
2. **Check logs**: Look for the new payload format in WooCommerce logs
3. **Test order confirmation**: Place a test order with Loadlink shipping
4. **Verify API calls**: Check that the API receives the correct payload format

The system now sends the exact payload format expected by the Loadlink API, without the unnecessary freight_id field.

