# WooCommerce Blocks Checkout Compatibility

## Overview

This document explains how the Loadlink order confirmation system has been enhanced to work with both traditional WooCommerce checkout and the new WooCommerce Blocks checkout.

## Key Differences Between Traditional and Blocks Checkout

### Traditional Checkout
- Uses shortcodes: `[woocommerce_checkout]`
- Processes orders via `woocommerce_checkout_order_processed` hook
- Session data is readily available during checkout
- Order confirmation happens immediately after order creation

### Blocks Checkout
- Uses Gutenberg blocks: `wp:woocommerce/checkout`
- Processes orders via Store API: `woocommerce_store_api_checkout_order_processed`
- Session data might not be available during checkout
- Order confirmation may need to be triggered via AJAX
- Different timing and data availability

## Implementation Details

### 1. Enhanced Hook Registration

The system now registers multiple hooks to ensure compatibility:

```php
// Traditional checkout hooks
add_action('woocommerce_checkout_order_processed', [__CLASS__, 'save_freight_data_to_order'], 5, 1);
add_action('woocommerce_checkout_order_processed', [__CLASS__, 'process_order_confirmation'], 10, 1);

// Blocks checkout hooks
add_action('woocommerce_store_api_checkout_order_processed', [__CLASS__, 'handle_blocks_checkout_order'], 10, 1);
add_action('woocommerce_store_api_checkout_update_order_meta', [__CLASS__, 'save_freight_data_blocks'], 5, 1);

// Additional coverage hooks
add_action('woocommerce_new_order', [__CLASS__, 'handle_new_order'], 10, 1);
add_action('woocommerce_payment_complete', [__CLASS__, 'handle_payment_complete'], 10, 1);
```

### 2. Blocks-Specific Methods

#### `handle_blocks_checkout_order($order)`
- Handles Blocks checkout order processing
- Saves freight data using Blocks-specific method
- Processes order confirmation

#### `save_freight_data_blocks($order_id)`
- Enhanced session data handling for Blocks checkout
- Fallback to API data if session is unavailable
- Marks orders as Blocks checkout for tracking

#### `handle_new_order($order_id)`
- Processes new orders created via Blocks checkout
- Checks for Blocks checkout marker
- Triggers confirmation if needed

#### `handle_payment_complete($order_id)`
- Handles payment completion for both checkout types
- Ensures confirmation is triggered after payment

### 3. JavaScript Enhancements

The JavaScript has been enhanced to detect and handle Blocks checkout:

```javascript
// Blocks checkout detection
function initBlocksCheckout() {
    const isBlocksCheckout = document.querySelector('.wc-block-checkout, .wp-block-woocommerce-checkout, .wc-block-components-checkout');
    
    if (isBlocksCheckout) {
        handleBlocksCheckoutConfirmation();
    }
}

// Order confirmation triggering
function triggerBlocksOrderConfirmation() {
    // Extract order ID from URL or page content
    // Send AJAX request to confirm order
}
```

### 4. AJAX Endpoints

New AJAX endpoints for Blocks checkout:

```php
// AJAX endpoint for manual order confirmation
add_action('wp_ajax_loadlink_confirm_order', [__CLASS__, 'ajax_confirm_order']);
add_action('wp_ajax_nopriv_loadlink_confirm_order', [__CLASS__, 'ajax_confirm_order']);
```

## Session Data Handling

### Traditional Checkout
- Session data is available during checkout
- Freight data is stored in session during rate calculation
- Data is retrieved from session during order processing

### Blocks Checkout
- Session data might not be available
- Fallback to API data retrieval
- Alternative data sources:
  - Static freight data from `Loadlink_Client`
  - Order meta data from previous calculations
  - API calls to retrieve current freight data

## Order Processing Flow

### Traditional Checkout Flow
1. User selects Loadlink shipping method
2. Freight data stored in session
3. Checkout completes → `woocommerce_checkout_order_processed` fires
4. Freight data saved to order meta
5. Order confirmation processed immediately

### Blocks Checkout Flow
1. User selects Loadlink shipping method
2. Freight data stored in session (may be cleared)
3. Checkout completes → `woocommerce_store_api_checkout_order_processed` fires
4. Freight data saved to order meta (with fallback to API)
5. Order confirmation processed
6. Additional triggers via `woocommerce_new_order` and `woocommerce_payment_complete`

## Testing and Debugging

### Test Scripts
- `test-blocks-compatibility.php` - Comprehensive compatibility test
- `debug-order-confirmation.php` - General debugging script

### Debugging Features
- Enhanced logging for Blocks checkout
- Separate log messages for traditional vs Blocks checkout
- AJAX request logging
- Session data availability checks

### Log Messages
```
// Traditional checkout
"save_freight_data_to_order called for order: 123"
"Order 123 uses Loadlink shipping, proceeding with freight data save"

// Blocks checkout
"save_freight_data_blocks called for order: 123"
"Order 123 uses Loadlink shipping (Blocks checkout), proceeding with freight data save"
"Retrieved freight data from API for Blocks checkout"
```

## Configuration

### Plugin Settings
The plugin automatically detects the checkout type and uses appropriate methods:

```php
// Check if Blocks checkout is being used
$is_blocks_checkout = $order->get_meta('_loadlink_blocks_checkout');
```

### JavaScript Configuration
Enhanced localization for Blocks checkout:

```php
wp_localize_script('loadlink-checkout-blocks', 'loadlinkAjax', [
    'ajaxUrl' => admin_url('admin-ajax.php'),
    'nonce' => wp_create_nonce('loadlink_set_dropoff_type'),
    'confirmNonce' => wp_create_nonce('loadlink_confirm_order'),
    'default' => get_option('loadlink_delivery_building_type', 'Residential'),
    'sessionValue' => WC()->session->get('loadlink_dropoff_building_type')
]);
```

## Troubleshooting

### Common Issues

1. **Session Data Not Available**
   - Solution: Use API fallback in `save_freight_data_blocks()`
   - Check: `Loadlink_Client::get_all_freight_data()`

2. **Hooks Not Firing**
   - Solution: Multiple hook registration for coverage
   - Check: `woocommerce_new_order` and `woocommerce_payment_complete`

3. **JavaScript Errors**
   - Solution: Enhanced error handling in `triggerBlocksOrderConfirmation()`
   - Check: Browser console for AJAX errors

4. **Order Confirmation Not Triggered**
   - Solution: AJAX endpoint for manual triggering
   - Check: `ajax_confirm_order()` method

### Debugging Steps

1. **Run Compatibility Test**
   ```bash
   # Access: https://your-site.com/test-blocks-compatibility.php
   ```

2. **Check Logs**
   - WooCommerce > Status > Logs
   - Look for Loadlink entries
   - Check for Blocks-specific messages

3. **Test Both Checkout Types**
   - Traditional: Disable blocks on checkout page
   - Blocks: Enable blocks on checkout page
   - Compare behavior and logs

4. **Monitor Network Requests**
   - Check browser developer tools
   - Look for AJAX requests to `loadlink_confirm_order`
   - Verify nonce and order ID

## Best Practices

### Development
- Always test with both checkout types
- Use enhanced logging for debugging
- Implement fallback mechanisms
- Test with HPOS enabled/disabled

### Production
- Monitor logs for Blocks-specific issues
- Use test orders to verify functionality
- Keep both traditional and Blocks checkout working
- Regular compatibility testing

## Future Considerations

### WooCommerce Updates
- Monitor for new Blocks checkout hooks
- Update compatibility as needed
- Test with new WooCommerce versions

### Performance
- Consider async processing for order confirmation
- Optimize API calls for Blocks checkout
- Implement caching for freight data

### User Experience
- Ensure seamless experience across checkout types
- Provide clear error messages
- Implement retry mechanisms for failed confirmations

## Conclusion

The enhanced Loadlink order confirmation system now provides full compatibility with both traditional WooCommerce checkout and the new Blocks checkout. The implementation includes:

- Multiple hook registration for comprehensive coverage
- Enhanced session data handling with API fallbacks
- JavaScript enhancements for Blocks checkout detection
- AJAX endpoints for manual order confirmation
- Comprehensive testing and debugging tools

This ensures that order confirmation works reliably regardless of the checkout method used by the customer.

